/**
 * jsPDFEditor
 * @return {[type]} [description]
 */
var jsPDFEditor = function() {

	var editor,demos = {
		'images.js': 'Images',
		'font-faces.js': 'Font faces, text alignment and rotation',
		'two-page.js': 'Two page Hello World',
		'circles.js': 'Circles',
		'font-size.js': 'Font sizes',
		'landscape.js': 'Landscape',
		'lines.js': 'Lines',
		'rectangles.js': 'Rectangles',
		'string-splitting.js': 'String Splitting',
		'text-colors.js': 'Text colors',
		'triangles.js': 'Triangles',
		'user-input.js': 'User input',
		'acroforms.js': 'AcroForms',
		'autoprint.js': 'Auto print',
		'japanese.js': 'Japanese'
	};

	var aceEditor = function() {
		editor = ace.edit("editor");
		editor.setTheme("ace/theme/github");
		editor.setOptions({
			fontFamily: "monospace",
			fontSize: "12px"
		});
		editor.getSession().setMode("ace/mode/javascript");
		editor.getSession().setUseWorker(false); // prevent "SecurityError: DOM Exception 18"

		var timeout;
		editor.getSession().on('change', function() {
			// Hacky workaround to disable auto refresh on user input
			if ($('#auto-refresh').is(':checked') && $('#template').val() != 'user-input.js') {
				if(timeout) clearTimeout(timeout);
				timeout = setTimeout(function() {
					jsPDFEditor.update();
				}, 200);
			}
		});
	};

	var populateDropdown = function() {
		var options = '';
		for (var demo in demos) {
			options += '<option value="' + demo + '">' + demos[demo] + '</option>';
		}
		$('#template').html(options).on('change', loadSelectedFile);
	};

	var loadSelectedFile = function() {
		if ($('#template').val() == 'user-input.js') {
			$('.controls .checkbox').hide();
			$('.controls .alert').show();
			jsPDFEditor.update(true);
		} else {
			$('.controls .checkbox').show();
			$('.controls .alert').hide();
		}

		$.get('examples/js/' + $('#template').val(), function(response) {
			editor.setValue(response);
			editor.gotoLine(0);

			// If autorefresh isn't on, then force it when we change examples
			if (! $('#auto-refresh').is(':checked')) {
				jsPDFEditor.update();
			}

		}, 'text').fail(function() {

			$('.template-picker').html('<p class="source">More examples in <b>examples/js/</b>. We can\'t load them in automatically because of local filesystem security precautions.</p>');

			// Fallback source code
			var source = "// You'll need to make your image into a Data URL\n";
			source += "// Use http://dataurl.net/#dataurlmaker\n";
			source += "var imgData = 'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQEASABIAAD/4QCqRXhpZgAATU0AKgAAAAgABgESAAMAAAABAAEAAAEaAAUAAAABAAAAVgEbAAUAAAABAAAAXgEoAAMAAAABAAIAAAExAAIAAAAeAAAAZodpAAQAAAABAAAAhAAAAAAAAABIAAAAAQAAAEgAAAABQWRvYmUgUGhvdG9zaG9wIENTNSBNYWNpbnRvc2gAAAKgAgAEAAAAAQAAA4CgAwAEAAAAAQAAA4AAAAAA/9sAQwADAgICAgIDAgICAwMDAwQHBAQEBAQIBgYFBwoJCgoKCQoJCwwQDQsMDwwJCg4TDg8QERISEgsNExUTERUQERIR/9sAQwEDAwMEBAQIBAQIEQsKCxERERERERERERERERERERERERERERERERERERERERERERERERERERERERERERERERER/8AAEQgDgAOAAwEiAAIRAQMRAf/EAB8AAAEFAQEBAQEBAAAAAAAAAAABAgMEBQYHCAkKC//EALUQAAIBAwMCBAMFBQQEAAABfQECAwAEEQUSITFBBhNRYQcicRQygZGhCCNCscEVUtHwJDNicoIJChYXGBkaJSYnKCkqNDU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6g4SFhoeIiYqSk5SVlpeYmZqio6Slpqeoqaqys7S1tre4ubrCw8TFxsfIycrS09TV1tfY2drh4uPk5ebn6Onq8fLz9PX29/j5+v/EAB8BAAMBAQEBAQEBAQEAAAAAAAABAgMEBQYHCAkKC//EALURAAIBAgQEAwQHBQQEAAECdwABAgMRBAUhMQYSQVEHYXETIjKBCBRCkaGxwQkjM1LwFWJy0QoWJDThJfEXGBkaJicoKSo1Njc4OTpDREVGR0hJSlNUVVZXWFlaY2RlZmdoaWpzdHV2d3h5eoKDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uLj5OXm5+jp6vLz9PX29/j5+v/aAAwDAQACEQMRAD8A/VOiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAK+MP+CnH7X2p/s1/Cm08KeANTFp478b+ZBZXKEGTTbRMCa5X0cllRD6lmHKV9n1/Pj/AMFHvjDJ8Y/2uPGt9BdmfSvC848L6YA25VjtCySlT0Ia4M7gjghx16kA+cNW1jVte1GfV9d1S71G+uWLz3V3O000rHqWdiST7k1f8P8Ajfxp4TdZfCvi/W9GdTlW0/UJbcg5zkFGHcVi0UAfQHgr9v39sbwE8Z0b4/8Aii8SMk7NZmTVVYHqD9qWQ4/Hjtivo/4ef8FqPjzoPlwfEb4deE/FsCY3SWjS6ZdP65cGSP8AKIV+eFFAH7efDH/gsR+y54xMVp45svE3gW7cgPJe2X220BPpLblnx7tEo/p9efDv4w/Cr4t2P9pfDL4ieHvE8IQSP/ZmoRTvED/fRTuQ+zAGv5h6uaRrOr+H9Rg1jQdVvNNv7Zt8N1aTtDNEfVXUgg/Q0Af1OUV+C/wY/wCCp/7WPwnkgs9c8VweP9HjIDWniWMzT7eMlbtSs27A4Ls4HXaa/QP4Ef8ABXL9nD4oG20j4jpe/DbW5sKf7Sb7RprNjot2gG0deZUjA45NAH3LRVLRta0bxHpdtrnh7VrLVNNvUEtteWc6TwTIf4kkUlWHuDV2gAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKAOb+JXjK3+Hfw58U/EC7VGh8M6Ld6vIrnCssELykH67K/mBvr271K9uNRv53nubqVp5pXOWd2JLMfckk1/QN/wUq8Vnwj+xT8SruKYpPqNpbaTEASC/2i6hicZH/TNpDz1AI71/PnQAUUUUAFFFFABRRRQAUUUUAeo/A/9pz44/s66uNU+E/j/UdJhZ99xpzP51hdevmWz5RiRxuwGHZhX6hfs3f8Fifhh44+y+G/2g9D/wCEH1l8R/2xZB7jSZm9WXmW3yfXzFHJLqK/GyigD+prRdc0XxLpVrr3h3V7LVdMvoxNa3tlcJPBOh6MkiEqw9wavV/OB+zn+2B8dP2X9YW8+GnitzpMkokvNB1DdPpt367osjYx/vxlX98cV+x37JP/AAUZ+C/7UC23hi7lHg3x46gNoWoXCmO7bv8AZJ+BN/uEK/X5SBuoA+saKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKAPgv/AILN6+dK/ZU0jSI3G/WvGNnbsucExpb3UpPTnDRoO3UfQ/iXX69f8FvtVMPw0+GOh7pMXeu3l3gY2HyoFXJ75/f8fU1+QtABRRRQAUUUUAFFFFABRRRQAUUUUAFOhmlt5UngleOWNg6OjEMrDkEEdDTaKAP0i/Yr/wCCsXiPwLJY/Db9py9u9e8OZWG08U7Wm1DTx0AuAMtcRj+8Myrz/rOAP108O+I/D/i/Q7HxP4V1qy1fSNShFxZ31lOs0FxGejI6khh9K/lpr6S/ZF/bt+Ln7JF/c2WhbPEfhG+DPc+HL+dkgWYg7ZoXAJhfP3tow44YEhWUA/oWor8MviN/wV6/a48ZtLD4VvvDfgi1bKqNJ0tZ59p/vSXRlGfdVX2xXzT46/aO+PvxMDx+PfjL4y1uCTk211rM7W4+kIbyx+C0Af0YeLfjV8HPALOnjn4r+DvDzxkhk1TXLa1YEHBG2Rwc5IGPU15F4g/4KMfsU+GXaPUfj9okxU4P9n2l3fjrjg28Lg1/PPRQB+6ur/8ABXf9jTTQxs9e8U6qVAIFpoMqlsnGB5pTkdef16VzN3/wWk/ZUtp2ih8HfE+6UYxLFpNiFP8A33eKf0r8UKKAP2Xn/wCC2X7PizOtt8LPiHJEGOxnisUYj1Ki4OD7ZNWLH/gtf+zXIjHUvht8TIGB+UQWdhMCPcm7XH61+MFFAH7a2P8AwWc/ZOu42efw38SLIhsBJ9ItCW9xsumGPxrrNL/4K0/sV6hJsu/G+u6aN6rvuvD90wwerfulc4HfjPoDX4PUUAf0MaH/AMFGP2KfEOz7B8ftEi39Pt1pd2XfHPnwpjn17c9Oa9H8P/tL/s6eKyqeG/jx8PtSkYZ8q38S2byjgnlBJuHCk8jsa/mgooA/qhs76y1G3W70+8guoH+7LDIHRvoRxU9fyy6Rr2u+H7j7XoOtX+mz/wDPWzuXhfv3Ug9z+deq+F/2yf2rPBxX+wf2hPHiIhysN1rU13EvOeI5mdRyTnjmgD+kWivwd8Jf8FZv20fDPlrqXjTQ/E0ceAE1fQrcZAGMFrcRMfXJOc969p8If8FvfiRaPEPH3wN8NaqvSVtH1KewP1USif64J9sjrQB+vNFfn/4H/wCCz/7NmvGK38aeD/Gnhad/vy/Zob61j/4HHIJD+EVfSHw+/bd/ZN+KDRxeEfjz4Ua4mIWO11G6Om3EhPZYroRux9gDQB7hRTIZoriJJ4JUkikUOjowKsDyCCOop9ABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAflf/wXMuJlt/gtarIRDI+uyMnYso04A/gGb86/Kev1G/4Lk3ssmtfB3TiqeVBa6xOpAO4s7WQOfb92P1r8uaACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigDu/hv8ePjR8H51m+GPxR8TeGlDb2g0/UpY7eQ5z88OfLcZ7MpFfXnwo/4LI/tI+DTFafErQvDvj+yQAPLLD/AGbfNj0lhHlD8YSfevgiigD90Pg9/wAFbv2V/iQYLDxjfat8PdUlIUprNv5tmW/2bmHcAP8AakWPp9M/YnhvxR4Z8Y6RD4g8I+ItM1zS7nmG9027jubeT/dkjJU/ga/lrrrvhv8AF74ofB/WBr3wv8fa54YvSQXfTrx4lmA6CRAdsi/7Lgj2oA/p6or8f/gP/wAFofiF4fNvo/7Qfge28V2S4V9Z0RUs9QA7s8BxBKfZfJFfpD8Cv2tPgB+0dZpJ8K/iJYX2obN82j3JNtqMPGW3W74ZgO7puT0Y0Aev0UUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQB+VX/AAXOAz8E2wMn+3wT/wCC2vysr9av+C4ukibwn8Jdd8sE2mo6pab9xyPNjtmxjoc+R17Y96/JWgAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAqxp+o6hpN9BqmlX1xZXlrIJoLi3laOWJwchlZSCpB6EVXooA+6/2b/wDgrZ8d/hTJbaD8XFPxJ8OKwVpbyXy9Wt17lbnB83HJxKGJ4G9RX6sfs+ftdfAb9pvTPtXwt8aQzalGm+60S+AttStfXdAT8wH9+Munbdmv5vKuaNrWs+HNVtdd8Pate6XqVjIJrW8sp3gngcdGSRSGUj1BzQB/U5RX4/8A7K3/AAWE8X+FDaeD/wBpvT5vE2kqBHH4k0+FV1KAdB58Q2pOo4+Zdr4BJ8wmvUfiZ/wW1+HmmSSWnwk+Dut68RlVvdbvY9PiB7MIoxKzj2LIfpQB+l9Ffh/4z/4LGftZeIpJF8N2vg3wrCeI/sWlNcSqM55a4kkUntwgHt3rx/X/APgoZ+2j4lLHUf2gfEMO/r/Z8dvY+nTyI0x07f1oA/ogor+ajU/2qf2nNYEi6n+0T8S7hJH3tG/iu+8vPsnm7R+ArN/4aF+Pv/RcPiB/4U17/wDHKAP6aaK/mW/4aF+Pv/RcPiB/4U17/wDHK2NO/az/AGpNKaI2P7R3xMRYBtjjbxXfPGoxjGxpSuPw4oA/pRor+dvRv+Cgv7Zug7fsP7QXiWXYMD7b5N53zz50bZ/H6dK9E8P/APBWr9tPRtv9o+NdB17aef7Q8P2qbuT1+zrF6449B75AP3gor8gfBX/Bbv4o2Uka/EX4KeFtYjzh20a+uNObHqBL54yOOOM+2ePpr4Y/8FgP2VfGzxWfjMeI/Al3I20vqVgbq1yemJbcu2O2WRQO/HNAH3JRXK+APip8NPirpZ1n4a+PdA8T2S43y6VqEVyIz6OEJKH2bBrqqACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigD4C/4LReHn1P9mDw9rsKFm0bxlbNIecLFJbXSHoP75j6kD8cV+Klf0Af8FPfC/wDwlH7E3xBEcYefShZapDkA7fKvIS56f88zJ0x+Wa/n/oAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigDU8M+K/FHgvWIfEHg/xHqeh6pbnMV7p13JbTx9+HQhh0Hevtf4Ef8Fef2ifhq9vpXxRt7H4kaJGQrNeYtNTjX/ZuY12v3P7yN2P94V8KUUAf0Mfs8f8FAf2a/2jvsuleGvGI0LxNc/KPD+vbbW7Z/7sTZMc2ecCNy2BkqK+kK/lXBKkMpIIOQR2r7R/Zd/4Kk/Hf4ENZ+GvHlzL8RPBsO2IWmpXBGoWcY4/cXZBYgDGEk3rhQqlM5oA/deivJv2e/2o/gz+054aPiD4VeKY7ueBFa/0m5Ah1CwJ7Swk5xngOpZDg4Y16zQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFAHnv7Q/hBvH/wABPiN4KjiEk2t+FtRsoBzxK9tIIzx3D7T+FfzM1/VRX8x3xx8EH4a/Gfx18P8AyhGnh3xFfabEAMAxxTuqED0KhSPYigDiKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKAOi+H/xD8b/CvxZYeOvh34mvtB13TH8y2vbOTa6+qkHh1I4ZGBVhkEEcV+73/BPv9smT9rz4Z6lc+JNMt9P8ZeEpobTXI7UEW9wsquYbiMHJQP5UgKZOGjOOCAP5/wCv13/4Ig+C9Q074efE/wCIMystnrur2OkwZ6FrOKWRyP8AwOUZ9j6GgD9MaKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAr8DP+Cp/hzQ/D/wC2n4zl0O+tZhq1vY6jdwQMCbW4a2jWRH9GYoJSP+mor9fP2z/2ntG/ZU+CWp+PZTbXHiG8/wCJf4c0+U5+1Xjg4ZlBBMcYzI/ThQuQWWv54PEviTXfGPiHUvFfifVJ9S1fWLqS+vrydt0k80jFndj6kkmgDNooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAsadp99q2oW2laZaS3V5ezJb28ESlnlkchVVQOSSSAB71/SL+yj8EoP2eP2fvBvwpCx/btKsBLqkiHIkvpiZbghu4Ejsqn+6qjtX5U/wDBIz9mh/ij8aZfjZ4j08SeG/h06yWnmLlbnVWGYQPXylJlJHIbyfWv2xoAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACmTTRW8TzzypHFGpd3dgFUDkkk9BT6+JP8Agqz+0w/wU+Ap+HXhu+EXij4kiXTEKN89tpwUC6l46Fg6xDP/AD0cjlKAPzO/4KDftS3H7T/x4v7/AEe/eTwX4XL6T4ciVj5ckYb95dY6bpmG7PB2LGD92vmOiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigArd8CeCPEvxJ8ZaL4B8H6c9/rWv3sdhY268b5HYAZPZR1JPAAJPArCr9dv+CQv7Ih8M6C/wC1H480tk1XW4ntPCcEyYaCzb5ZLvB6GXlEOB+7DEZEooA+5P2bfgT4b/Zv+Dfh74TeGysy6VBvv7zZta+vH+aacj/afOAc7VCrk7a9OoooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACv53/29/2gm/aN/aV8S+LbC9M/h3R3/sLw/hsobSBmHmLjqJZDJL64kA7V+xH/AAUU+OT/AAI/ZW8V6zp159n1zxIg8NaOwbDrNcqwd1PUMkKzOD2ZVr+e+gAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAoorrPhT8LvGPxo+IWifDLwFpjX2t67crb26chIx1eR2/hRFBZm7BTQB7j+wF+yNfftW/GKGz1m1mTwJ4YMd94lulJTzUJPl2iMOQ8pUjIIIRZGByFB/oD07T7DSNPttK0qygs7KyhS3treCMJHDGgCqiqOFUAAADgAV5j+zJ+zz4P/Zi+EWkfC3wkizNbD7RqmoGMLJqN44HmTP8AXAVRztRVXJxmvVqACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooqG7u7awtZr68nSG3t42llkc4VEUZJJ9ABmgD8cP+Cz/xlPij4z+G/gvp1wDZ+CNN+3Xyg/8AL7dhXCsO+2BISP8Ars1fnZXd/Hj4m3Xxl+M/jT4pXW8f8JLrVxfQoxJMUDORDHz/AHYwi/8AAa4SgAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigB8EE91PHbW0LyzSuEjjRSzOxOAAByST2r90f+CbH7Eqfs1eAT8QfH+moPiP4stlN1HIoLaPaHDLag9nPDSEfxBV5CZb5m/4JFfsf+FvGBk/ai8cTWWqf2JqL2Hh3S1cOLa7jCs9zOvZ1Dp5anpkuR/qzX620AFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAV85/wDBQr4nt8KP2QfiJrltP5d9qunf2BZYOGMl4wt2Kn1WOSR/+AV9GV+ZP/Bbr4ifYvA3w3+FNvOd2rapc69dIpxhbeMQxbvUMbqTHuh9qAPyNooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiirOmaXqet6hb6To2n3N9fXTiOC2tomkllY9FVVBJPsKAK1T2NhfandxafptnPd3U7bIoIIzJJIfRVHJP0r69+Cn/BPTxR4lFvrPxb1CbRLWTDrpFjtkvZB1w78rF9AGbsdpr7q+F37NnhX4a2ItPAXgKx0YMu2S6kGbmX/flfMjD2JwOwFcNbH04aR1Z6NDLatTWWi/E/M/wD+w/wDHvxukV3e6Bb+GbKXnzdamMMmP+uKhpAfZlWvffCn/AATZ8IWyxyeNviLq2oP1eLTLaO1X6bn8wke+B+Fff9t8NZjg3mqIvqIoy36nH8q0Ivhzoyj97dXbn2ZQP5V51TMaktnY9WnlVKO6v6nx5pP7C/7N2moFuvB97qbDHz3erXIOfXEboP0rb/4Y/wD2bv8Aoltj/wCBdz/8cr6t/wCFfaBt25us56+YM/ypknw60Rh8lxeIf99SP/Qa5/rc39pnQsFTW0V9x8kah+xT+zTqAJPw5Fu5AG+31S8TH4ebt/SuG1//AIJ1fBnUVZ9D1/xNpEvYfaIp4hx/daPd/wCPV9xz/DWMgm21ZgeweLP6g1mXPw91yHJge3uB2CuVP6jH61ccXUW0mTLA03vBH5qeLv8Agm948sBJN4J8e6PrCLkrFfQSWUpHoCvmKT9So+nSvBPHP7Ovxr+HKyTeKvh3qsNrFy13bRi6twPUyxFlX/gRBr9jrvQdZscm602dAOrBNy/mOKoV1U8wqL4tTiqZZSfw3R+HVFfr38Q/2cfgv8T1lk8U+BNPN7NydQs0+y3W71MkeC/0bcPavl74j/8ABOC9h82++FPjdLhRyun62ux8eguIxgn0BRR6mu2nj6Uvi0OCrltaGsdT4lortviF8FPin8LJmTx14K1HTYQ20XewS2rnjGJkJQnkcZzz0ria7IyUldM4JRcXaSsFFFFMQUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFAH6Q/8EXfjc/h34oeJ/gRql2RY+LrP+19LjY8Le2w/eKo9XgJY/wDXutfsPX8x3wP+Jt/8GfjB4P8Ainp29pfDOsW9+8a9ZolcebH9HjLofZq/pp07ULLVtPttV026jubO9hS4t5ozlJY3AZWB7ggg0AWKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAr8Nv8AgsF48Pir9rmTwxHMTD4N8P2WmFA2VEsoa7ZsepW5jB/3RX7k1/Nv+2F4y/4T/wDal+KfilZhLDceKb23t5A24PBDKYIiD6GOJP8A69AHj9FFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRX1b+xh+xBrf7Q19H448bfadJ+H9lNgyKpWbV3U/NFCf4UGCGl5wflXLZKZ1asaceeb0NaNGdWahBXZ5r+zt+yt8U/2ktZa38H6cLPRLSQR6hrt4pW0tjwSoPWSTBB2LzyCdoOa/VX4CfsZfCf4D6ag0qzbUdYkQLd6rcKPtE3qobqicfdXaOmcnmvZ/CnhPw14G8PWPhPwfolppGj6bEIbWztYwkca9enckkkk5JJJJJJNa1fPYnHVKzstEfUYTLqdBXesv62K9np1hp6bLK0ihHQ7FAJ+p6mrFFFcR6AUUUUgCiiigAooooAKo3uiaRqOTeafDIx6ttw35jmr1FMLXORvvh1p02WsLuW3Y9Fb51/of1rnb/wAD6/ZZaOBbpB3hbJ/I816hRVKpJEOnFnh95Z5WSyv7TKuCskU0fBB6gqe1eF/Eb9i/4D/EIS3Mfhk+G9Qk5+1aIwtxn3hwYjz1woJ9a+3buxs75PLvLWKZfR1BxXO6h8PdJuctYyy2jnoM70/I8/rW1PEOLvF2OerhYzVpK5+UPxE/4J4fFDw95l34A1vT/FVsMkQORZXeOww7GNvrvH0r5w8XfD/xx4BvPsHjTwnquizbiqi8tXjV/wDdYjaw91JFfuRqHgTXbPLwxpdIO8TfN+R/pmuY1XSLa9t5dK1zS4p4JRtltruAMjD0ZGGD+Ir0aWYzXxanl1sqg/hdj8P6K/VLxx+xZ+z74282ZfCB8P3cmf3+iTG2C/SLBiH/AHxXgXjP/gmzrERef4ffEa0uV6rbavbNCR7ebHuB/wC+BXbDHUpb6HBUy6vHZXPiqivafFn7HP7RHhIPLN8PrjVIF6TaTNHd7vpGh8z80ryfWvDfiLw3cfZPEWg6jpc/Tyr21kgf8nANdMakJfC7nHOnOHxKxnUUUVZAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABX9Bv/AATd+J5+KX7HfgK9uLgy33h62fw3eZOSptG8uIE+8HkH/gX41/PlX6x/8EQfiC82g/E34VXE52Wl3aeIbOPPUyo0E7f+QLcfjQB+otFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQBl+KtetvCvhjV/E95j7Po9hPfy5OBsijZzz24Wv5cb69utSvbjUb6YzXN1K080h6u7Eljx6kmv6N/21vE3/CI/slfFrWRL5Tt4TvbKN920q9xGbdSDkYO6UY5644PQ/zhUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFerfsz/AHxB+0d8U7DwFpLva2CD7XrGoBciytFI3t6FySFUd2YZwASJnJRi5S2RUISnJQjuz1P8AYf8A2N739ojxG3i/xpBc2nw/0aYLcOpKPqk4wfs0bdQoHLuOQCFGC2V/YDR9H0rw/pVpoeh6db2GnWEK29ra28YjihjUYVVUcAADGKz/AAT4K8M/DrwnpfgnwdpUOnaPo9utta28Y4VR3J6sxOWZjyxJJJJNblfMYrEyrzu9uh9hg8JHDwst3uwooorlOsKKKKACiiigAooooAKKKKACiiigAooooAKKKKACorm0tbyPyru2imT0kUMP1qWigDmr7wDod1lrcS2rn+42V/I/0xXP3vw71WDLWVxDcqOgPyMfwPH616LRVqckQ6cWePXehazY5+1abcIB1bYSv5jisu6tLW+ge1vbaK4hkGHjlQOrfUHg17rVW50vTbzJutPt5Se7xgn86tVe5Do9mfLGv/s+fA/xOXbWfhV4Zkkkzvli0+OCVs+rxhWz75rzzXP2EP2dNX3Gz8Oano7Nzmx1SY4PsJS4H5Yr7Yn8FeG5jn7B5ZPdJGH6ZxVKX4daK/Mdxdxn/fUj+VbRxU1tJnPLBwlvFM/PTWv+CbHgKfd/wj3xI1+yz937bbQ3WPrsEea4jVf+Cavi+En+xPiho92M8fa7CW3yM/7LSdq/TaT4aQn/AFOruv8AvQg/1FVZPhtfj/ValA3+8rL/AI1tHMKq+0c8ssov7P4n5Ral/wAE9vj7Y5+yzeF9Rx/z7ai654/6aRp9K5e//Yo/aWsMn/hXP2hBgb7fVLN85/2fN3fpX69yfDzXk+7JaP8A7sh/qBVd/A3iVPu2SPz/AAzJ/U1qsyqeRi8pp+Z+Nd5+zB+0FYnE3wl8Qt8xX9zbebyP9wnj3rn734N/F7TRnUfhX4vthgtmbQ7lBgdTkp0r9rH8JeI0+9pUp+hU/wAjVd9A1xPvaPeevEDH+laLM5dUjN5THo2fhxqGiazpJxqukXtkcgYuLd4+SMgfMB2qlX7lzabeBGS4sJgjDDB4jgg+uRXJa18LPhp4h3r4g+HfhvUC3DG60qCRvzZcg+9aLMl1j+JjLKX0l+B+MNFfq54g/Y4/Zy8Qhmm+HFtZStnEmn3U9ttz6KjhPzU15h4m/wCCcHwxvxJJ4V8a+IdIlflVuVivIk+i7UbH1c1tHMKT3ujCWWVltZn55UV9J/E79gz4xeBrebVPDRtPF+nwKXb7ADHdhR3+zty30RnPtXzfNDNbTSW9xE8UsTFJI3UqyMDggg8gg9q64VITV4O5x1KU6btNWGUUUVZmFFFFABX23/wSA8aN4Z/bBtfD5lcJ4u8PX+l7AflZo1W8BI9QLVuevJ9TXxJXt/7EHiY+Ev2u/hJq4baJPFVnp7NxgLcv9mJOe2JjQB/RzRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAfJH/BVTXv7F/Yj8b2yyFJNXutNsEIJB5vYJGHHqsTA+xNfgdX7d/8FldU+wfsk6faea6f2l4ysbXCjh8QXUuD7fus/UCvxEoAKKKKACiiigAooooAKKKKACiiigAooooAdFFJNIkMMbSSSMFRFGSxPAAHc1+2P7E/7OVv+zx8H7Sz1SzRPFviJU1HXpcDejlf3dtn0iViMcjeZCODX58/8E4/gZH8WfjpF4q1m087QfAax6tOGXKS3ZYi1jP/AAJWk9/JI71+xFeLmmI1VGPzPoMnw2jry9F+oUUUV4x7oUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAU2SKOUbZY1cejDNOooAozaHo1xnztKtWJ7+UoP54zWdc+BvDtwDttHgY94pCP0OR+lb9FNSYnFPocLffDdwC2m6iG9EmXH/AI8P8K+Vf2rf2MtP+KenXHiHS9Mh0bxlCmYb5VAg1DA4jnK8EkDAf7y8ZyBivuSmyRpKhjlRXRhgqwyD+FbUsROnLmizCthqdWLjJH87OuaJq3hrWLzQNe0+ax1HT5mt7m3mXDxOpwQapV+nP/BSf9lKw1fwrJ8ffAem+Xqmhoq+ILaJeLiz6CfH96LgH/YJOQIwK/MavpcNXjWp86Pk8VhpYeo4S+QUUUVucwV0fw21tvDPxF8LeJFcIdK1qzvgxYLt8uZHzk8D7vU1zlFAH9VFFUdCvZtS0TT9RuAolurWKZ9owNzICce2TV6gAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooA+AP+C1f/JrHhb/ALKBZ/8ApBqNfitX7U/8Fq/+TWPC3/ZQLP8A9INRr8VqACiiigAooooAKKKKACiiigAooooAKKK3PAvhW88deNvD/grTywudf1O30yIgZIaaRYwce27NJuyuxpNuyP19/wCCd3woT4afs2aLqd1bCPU/GbnxBcsR8xikAFuM+nkqjD3kb1r6cqrpOl2OiaVZ6LpkCwWen28drbxL0SNFCqo+gAFWq+Rq1HUm5vqfcUaap01TXQKKKKzNAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigCtqWm2Gs6ddaRqtpHdWV9A9tcwSrlJY3UqysO4IJB+tfg5+0b8Irr4GfGfxP8NphI1rpt2ZNOlfkzWkgEkDZ7nYyhsfxBh2r97K/Or/grX8Lo2tfBnxlsrfEkbv4c1FwM7lIae3+mCLgZ/2l9K9LLK3JV5Hszys3oc9H2i3j+R+b9FFFfQny4UUUUAf0/fCl3k+F3g6SRizNoFiWYnJJ8hOa6quZ+GNrPY/DbwnZXUeya30OyikXIO1lgQEZHB5FdNQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAfD3/BYfTHv/ANkA3SgEab4psLpsrnAKzRcHtzKOfw71+HFf0C/8FNvDzeIv2I/iRDEhaaxistQj68CK9t3c/wDfAev5+qACiiigAooooAKKKKACiiigAooooAK+hv2AfDK+KP2svA0M0YaDTpbjU5MjO0w28rof+/gSvnmvs7/glPoo1D9orV9UkjBTS/CtzIrccSPPbRgevKs/I9PeufFS5aMn5HTg482IgvNH6z0UUV8ofaBRRRQAUUUUAFFFFABRRRQAUVzGv/EPw/oZaFZvttyOPKgIIU/7TdB+p9q8o8XfH42jPDLrFtpoH/LG3/eTfQnBIPvgVpGnJi5ru0dWe+SSRxKZJZFRR1LHAFUJPEfh6JtkuvacjejXSA/zr491j45W1zK0kdnf38n9+5m25/8AQjWp8KvG7ePPH2l+FNU017a11N2gSS2JaRZNpKDkYOSAvTuK1jQTajfcudKtGDqOOiVz6vi8Q6BM22HXNPkb0W5Qn+dX0dJFDowZT0IOQa55v2dEkBMA1hOON4Q8/wDfIqufgD4t0wmfQ9Zv4HzkL5JXP1Kt/SumWWVVseTHNqD3Z1VFcobL4t+F+NW0VdXtl6sgKSgfiBn8j9av6H4w0bXG+zxSPbXY+9bXC7JB9PX8K46uHqU/jR3UsRTqq8Hc3KKKKxNgooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAK8M/be8CD4g/su+PdKSIPc6dp39tW5xlla0YTtt9ykbr/wI17nVPWdKtNd0e+0O/Tfa6jbSWk6+qOpVh+RNXTnyTUl0IqwU4OD6o/nWoq3q+mXOi6te6NeAC4sLiS2lAPG9GKn9QaqV9gfChSojyOscalmY4VQMkn0pK6n4UaOfEXxS8HeHxnOp6/Y2fDbf9ZcInXt1oA/p10qxGl6XZ6YshkFpAkAcjG7aoXOPwq1RRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAedftHeEj48/Z/wDiT4Njj3zax4U1K0gHJxK1tIIzgdcPtOPav5nK/qor+ZD47eA3+F/xq8dfDtojGnh3xDe6dDxgNFHM6xsPZkCsPYigDhqKKKACiiigAooooAKKKKACiiigAr70/wCCR8MTfEzx5clAZU0KFFbuA04JH47R+VfBdfe//BI3/kovj7/sC2//AKONcmO/3eR25d/vUP66H6fUUUV8ufYBRRRQAUUUUAFFZ974h0LTkMl7q1rEF7GQFvwA5NeU+Pv2iNA0KOS10q5USjjcQHlP+7H2+rY+lXGDYLV2WrPUNf8AE+j+HIPN1G5AdhlIU5kf6D+p4rwT4k/tAxxGSxS5KDkfYrR8uf8Aro/b6foa8i8T/Efxb4unl+yvLZQSn55XcmaQe7dR9B+Zrm7bRraI75iZn6/N0/KtEox82ehQy2pU1q6LsXtY8c+K/E7NFbu1nanjy4CVBH+0/U/y9qyoNBX711MSfRP8a1gAAAAAB2FFJzbPYpYWnTVoogisLOD/AFdumfUjJ/Wtjw3rEvh3xDpevwFhJpl7DeJt65jcOMflWdRU3d7mzhFpxtoz9dEdJEWSNwyMNysDkEeoNOrL8LSTTeGNIluCTK9hAz5GDuMa54+talfXJ3Vz8WkrNoK8B1XQtH1GaQXmnQSEOSG2YYHPYjkV76zBFLscBRk14c7F2Lnqxya8fNnbk+f6HuZIvj+X6le0tTZx+SLmaVB93zW3Mo9N3U/jk+9T0UV4p74UUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFAH4J/tLaQmhftDfErS4gBFD4r1Exjn5Ua4dlHPoGArzavcP23LWWz/as+JEMxXc2rCUYOeHijcfowrw+vr6LvTi/JHw9dWqyXmwr3T9hfw6fFP7YHwk0wIG8nxRa6jgnH/Hq32nPUdPJz/Q9D4XX2h/wSL8LN4g/bM0jVhEWHhnQtR1QkA/JuiFpk4/6+8c+v0rQyP3YooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKK8q/ah+PWn/sz/AAQ8RfGPUNBn1v8AsVYY4NPil8o3E00qQxq0hB2LukBZsHABwCcA/mLrv/Bbb46XBb/hGvhH4EsAT8v25ry7xz32SxZ447c8+1AH7H0V+G19/wAFbf21fE87QaBP4ZsJSu7y9L8PecygHJOJWkPcDn2781Xf9qf/AIKreP8AKaZc/E+8DkqF0bwOsRGDuIBgtAcj65x14oA/dGivwvPhT/grj45b95N8eIhMxUpJq11piHf1BUyRjb7EYHI4o/4YR/4KbeNVL+INH8USRSpndq/jm2fdtPClDdswPcZAHH0oA/ce+1LTtMjWXUr+2tEdtqtPKsYJ9ASetctqHxo+DukjdqvxY8G2Q37M3Gu2sfzenzOOeOlfjVZ/8Egv2yNYlkudT/4Q6zlchme911nZyepJjjfJHfPrxmupsf8Agif+0hIH/tP4l/DW3II2CC7v5s+uc2q4/WgD9RdR/a3/AGWNJEn279o/4Zq0T+W8aeKrKSRWzggoshbg9eOK5+8/bw/Y7sZjBN+0P4OZgAcw3vmr/wB9ICP1r4Dsf+CH/wAQJC/9p/Hzw9bgY2GDRp5s+ucyLj9a6Gw/4IaDML6n+0yT0M0UHg/8wrm9/Ur+FAH15cf8FJv2I7WOSWT496YyxglhHpl/Ixx6BYCT+Gay5P8AgqT+wqkbOvxvaQqCQi+GdXy3sM2oGfqa+dbL/gh74FjkY6l8f9enQj5RBocMRB9STK2fyrWsf+CI/wAGIy/9p/GTxpODjZ5FvaQ49c5Rs/pQB7B/w9g/Yl/6KPqv/hO33/xqoL3/AIK0/sV2sPmweN9dvG3Y8uHw/dBvr86qMfjXmtv/AMETf2elmRrr4p/ESSEH51jlsUYj2Y25A/I1pf8ADlT9lj/ofvir/wCDXTv/AJBoA6r/AIe8/sbf9BrxX/4IpP8A4qprT/grl+xjcziKbxP4ltVOcyzaBMVH/fG4/pXH/wDDlT9lj/ofvir/AODXTv8A5Bo/4cqfssf9D98Vf/Brp3/yDQB3/wDw9g/Yl/6KPqv/AITt9/8AGqu2P/BVH9hq7jZ5/jDc2RDYCT+GtUJb3Gy3YY/GvKr7/gij+zTJGo034kfEyBw3zNPe2EoI9ABaLg++az5/+CJXwJaF1tvi348SUg7GcWbqD6kCEZHtkUAe92X/AAUv/Ye1CIzQfHiyVVbaRPo+pQtng/de3BI564xW/Yft8/sb6i0K2/7QvhNDOAy+fcPABxn5i6jb9Dj0r5MvP+CIHw5eErYfHjxJDLkYabSIJVx/uh1P61h3/wDwQ1sXkDaZ+0xPCgXlJ/CAlJbnncLxcDpxg0AfemmftbfstauIxYftG/DR3lfYkb+KbKORj6BGkDfpX43f8FTtH8JL+1ZqHjnwN4l0bXNJ8aaVaar9p0q+iuoUnRTayoWjYgN/o6uR/wBNBXu1/wD8EPvH8ez+zPj54fuM53+fos8O3pjGJHz39Pxrwf8AaX/4JmfGz9mX4Z3XxZ8ReKvCeu6Hp9xBb3q6XLc/aIPNcRo5WSFV2b2VSd2cuvB7AHyHRRRQAUUUUAFFFFABRRRQAUUUUAFfcf8AwSW1TyfjZ4u0beQLvws1ztwMExXUC9euf3x4+vpXw5X1F/wTY8RpoP7V2gWUj7V13T77TSSQBnyWmAP1MAH1IrmxkeahNeR1YGXLiYPzP2Tooor5U+zCiiigAooooAqXGk6XdyebdadbSyf33iUt+fWvh/xtbW8PjfxB5UEaY1W56KB/y1avuuviL4mWL6d8QvEdrICP+JnPIuTk7Xcuuffawq09D1Mqt7SXoczRRRQe4FFFFABVjT7KfUr+2062GZrqZIIx6sxAH6mq9en/ALNXhJvGHxo8N2bRb7fT7n+1Lg4yAsHzjPsXCL/wKrpw55qK6mOJrKjRnVl9lN/cfpHbwJa28VtHnZEgRc9cAYqSiivrT8YKGvXItNFvbjOCsDY+pGB+prxuvSviHei30RbQH5rqULj/AGV+Y/qB+dea14GazvVUeyPpsmp8tFzfV/kFFFFeWeuFFFFABRRRQAUUUUAFFFFAEc/n+U32bZ5gGV352n2OOn1/nWZZ+JbGa6/s6+Bsb4HHkTHAY/7DdG/n7Vr1k+ItD03WLKT7eqL5alvNIztA559qat1E79DWor5O8JftKa34fvnsNdiP2Eyt5BYGRFTPAP8AEMD0J+le8+Gfi94T8RW8cpultvM6PuDxE+gcdPxAq3Te61NKkJ03aorHcUUyGeC5iWa3mSWNhlXRgyn6EU+syAooooAKKKKACiiigAooooA/Dr9t27kvf2rfiRNKqhl1YQjaOMJDGg/HCivD69H/AGk9ZTX/ANoT4k6vEQ0U/irUfKI7otw6qfxUCvOK+voq1OK8kfD13erJ+bCv07/4If8Ag77T4w+KXxBkhA/s/TbHRoZDn5vPkklkA7cfZY8/Va/MSv24/wCCN3gU+G/2WL7xdPEBN4u8S3V1G+PvQQJHbqPwkim/OtDI+8KKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKAMzxL4Z8OeMtCvPDHi7QdP1rR9Rj8q7sNQtkuLedMg4eNwVYZAPI6gGuP0P9nX9n7wyoXw78Dfh/pmO9p4asoicgAklYwSSAOe+K9DooAr2VhY6dD9n0+ygtYi27ZDGEXPrgDrViiigAooooAKKKKACiiigAooooAKKKhuru1sYGur25it4UxuklcIq5OBkngckCgCaiuI1n45/BPw4GPiH4w+CNLCbi32zxBaQbdv3s7pBjHf0ritW/bZ/ZG0UuLz9o3wDJsIB+ya3DdZyM8eUWz+HTvQB7ZRXzLqf/BSv9iDSd32r486e+2QxH7NpOo3PPPTy7dsjj73TpzyK5fUv+Csf7E9iJDa/EHWNQ8ttqi28PXq+YM9R5kacd+cGgD7Cor4evv+CxH7IFpIqQf8JveAjJeDREAX2O+VT+lYtz/wWl/ZVgneKLwX8ULhVPEsWk2AVvpuvQfzAoA++64j43fDHTvjP8IvF/wr1R0jh8T6RPp6ysMiCVlPlS49UkCOPda+Kb3/AILY/s6JNt0/4YfEeeLH3prexibPptFy3Hvmq/8Aw+0+AX/RKPiB/wB82X/x+gD8etd0TVPDWt6h4c1yze01LSrqWyvLd/vQzRuUdD7hlI/CqNex/tc/FP4bfG347+Ifiv8AC7w9q+h6b4lMd7d2OpJErpeFdszr5bsCrlRISTnc78dK8coAKKKKACiiigAooooAKKKKACu8+AvjRPh38a/A/jaaUR2+ka9aXF0xOP3HmqJRntmMuK4OilKKkmmVGTjJSXQ/o1oryr9ln4kL8WP2fvBHjV7kT3dxpUdrftnn7VB+5mJHbLxs2PRh616rXx84uMnF9D7mE1OKmuoUUUVJQUUUUAFfMH7T3hd9K8aWfiWKPFvr1oCzAYHnQ4jdR/wDymP+/X0/WD8Rvhyfir8PtS8NWManWtPYappOcAvIo2yRZ/21OPTcFJ6VrSjzPlXU2oYlYeoqkttn6P8Ay3PhqinzQzW8z29xE8UsTFHR1KsrA4IIPQg0ypPqgooooAK+2v2Ivho+h+F774kalCVudeP2axDDlbZG+Zv+BuPyjU96+cfgN8F9W+Mfi6OxVJIdDsWWXVbwDARM8Rqf774IHpye1fpDp2nWOkafbaVplsltaWcK28EKDCxooCqo9gABXq5bh25e2lstj4/inM4xp/Uqb1e/ku3z/L1LNFFZ2v6qmjaXNfMRvA2xKf4nPQf1+gNezKSjFylsj4SEHOSjHdnBePtT+3a0bWNsx2a+X/wI8t/QfhXNU6SR5ZGlkYs7ksxPUk02vk61R1Kjm+p9tQpKlTjTXQKKKKyNQooooAKKKKACiiigAooooAK5v4k37ab4A8Q3iOVdNOmVGHZmQqD+ZFdJXHfGD/kmXiL/AK8z/MUzSir1Ip90fFUsMU6GOVAynsaqW39taBObvQL+WIn7yg9fqDw341eoq1JrY+pqUozVpI6vwp8d9V0eUJqS3FrIDhprNioP+9GTg/54r2vwr+0Ampqo8+y1MY5VW8mYfVcf+y18yT2lvcjE0Qb37j8azpdCKt5lpcFSDkBu34itOeL+JHlVsqi9abt/XY+69N+KHhe+wtxLNZOe0yZXP1XP64rpbPU9O1Fd9hf29wOuYpA38q+ALHxV450XCxX808Y/hlxMMfjyPwxXuvwL07Xvipp2q6jeyxaYdNljhhkjiYiVyGLDlsrgbPX71VCgqkuWG55OLpTwsHVq/Cj6Torz5PBvxO0r/kHeJop0H3VeZm/R1IH51ahm+LtqQsmj2F6O5Mkan9HX+VVLA1l9k4Y5hh39pHb0VzCa147t4mlv/A6lEG5mjv4+B9MkmtDStY1XUGXz/D01oh6vJKvH4dTWE6M4fGrHRTrU6msHc16zvEetW3hrw/qniO9x9n0qzmvZsttGyNC7c9uFPNaNeJ/tp+Lf+EL/AGWviLqwmEb3OjtpaHPJN0623HviYn2xntU04801HuOrPkg59kfh7qF9dapf3Op3spkuLuZ55nJyWdiWY/mTUFFFfYHwoV/SP+x/8PD8K/2X/hl4HkgMFzZeHba4vIyuNlzOv2icfhLM/PfrX8/f7PHw2k+MHxz8CfDNYi8XiDXbW0uv9m38wNO34RK5/Cv6Y0RI0WONQqqMKoGAB6UAOooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKK+PPit/wVW/ZN+Fmu6r4X/tbxF4n1TR7mSyuodD0vfGsyEq6iWZ40YBgRuUkcHGaAPsOivyz8Z/8FwtOQvB8PfgBczAk7LrWdcWLHpmGKJs/9/PbnrXh3if/AIK//tgeMbk2Hg/T/CPh15jsgTStFe6uPX/l4klDNj0QD2oA/b2q2oalp2k2r32q39tZW0fLzXEqxov1ZiAK/DH/AITL/gq/8dMyWdx8aZbe5z+9sLKbRbVw3JAeJIY8e2cYOOhq1p3/AAS9/bz+KF1HqPjqxs9PmlO77T4m8UJcyAN1Y+S07joMgjPTjjgA/W7xZ+1/+y34ILp4k+P/AIFgmjzvgh1qC5nXHHMUTM4/L19DXjvir/gq/wDsV+HC6WHj7V/EMicFNK0K6688Bp0jU9OoOORzXyR4T/4Ig/EW78pvHXx18OaXwDKuk6VPf445AMjQZ54yR747V7L4W/4In/ASwCP4w+KXjnWZF5K2X2Wxjbp1BilbHXow7c8cgFLxN/wW4+DNoWHg74N+M9Vx906lcWtgD0/uNN7/AJD148l8Sf8ABb/4k3W//hEPgT4a0zP3f7S1W4vsfXy1hz+lfYvhj/glh+xP4bw9x8LbvW5l6Saprl6+Ov8AAkiIevdT0FeteGf2Rv2XfCAU+H/2ffAEEiHKzyaBbTzD6SSIz/rQB+Smrf8ABXb9s7xTcfZvD48J6TM+SsWk6A0zYHJwJ5JT0B/M1TH7QH/BV74oDGmT/Fy8jl+XzdH8JGzjHVc77e2QL0PORyD3r9xNJ0LRNAt/smhaNY6dBwPLtLdIU46cKAO5q9QB+Fn/AAzz/wAFXfiYu7UoPi5eJP8AN5eseMDaoM4fGy4ukCjOOMDBGMZGKmtv+CT37b/iucXPiOx0G0m5Hmar4kSdgOvWPzDySfxz9a/c2igD8ZNH/wCCJ/7RFwVOvfE74d2KkjP2ae9uWAxzwbdBn2zj3rtdJ/4IceI5lB139o3TbM7SSLTwzJcgNngfNcx8Y7/p3r9ZqKAPzI0//gh74Giz/a37QGu3PygD7PocMHPc/NK/Ht+tdRYf8ETv2d43B1T4ofEa4Tbgi3nsYSW9ctbPx14x+NfofRQB8H2n/BGb9ky2gEU2vfEa6cE5lm1i1DH/AL4tVH6VtJ/wSF/Y3VVU6R4scgYLHXXyffhcV9rUUAfHsf8AwSd/YmSNUb4e6vIVABdvEV7lvc4kAz9BTv8Ah0/+xL/0TnVf/Civv/jtfYFFAH57/tGf8EnvgAnwW8VXvwL8HalYeOLCyN9pO/Vrq5Fw8RDtB5cjlS0iKyKezMpr8XyCpKsCCDgg9q/qor8Kv+Cpf7L0nwK+O03j/wANaYYvB3xEkk1K2MaYitL/ADm5t+OFBZhKo4G2QqM+WaAPiyiiigAooooAKKKKACiiigAooooA/SL/AIJNfFyOSy8WfBHUrnEsLjxDpSsfvI2yK4UemCIWA/2nPY1+idfgV8Avi1qPwP8Ai74b+JmniSRNJux9sgRsG4tXBSaP0yUZsZ4DbT2r96NF1nS/EWj2PiDRL2K807UraO7tLiJtyTROoZGU9wQQa+ezOjyVedbM+oyivz0fZveP5F2iiivNPVCiiigAq1peoTaXqEF/B96F92P7w7j8RkVVopxk4u6FKKknF7M5/wCPf7L1h8UDJ8QPh1PDa65dJ5txayELBfnH3gf+WcvYk8E9cHLH448U+CPF/gm9Nh4t8N6hpUwYqPtMDKr4/uv91x7qSK+99J17U9Fk32NyVUnLRtyjfUf5NdxpPxB0y82xakhs5Txu+9Gfx6j8fzr0r4fEO7fJL8DLD5jjsBD2fL7WC26NeXX8j8uba1ub24S0s7eWeeU7UjiQs7H0AHJr3P4Ufsi/ELxzcxX/AIttZvDGighna6jxdzDuEiPK/wC8+AM5AbpX31DPDcRia3lSWNujIwIP4ipK7KWVwTvOVzmxXFtecXCjBQfe93+S/UwfBPgfwz8PfD9v4Z8KaZHZ2VuMnAy8r4GXdurMcck+w4AAreopCQoLMQAOST2r00klZHyc5ynJym7tgSFBZiABySe1eW+MfEP9t3/lW75tLYlY/wDbPdv8PatLxj4xW8V9K0qTMJ4mmB+/7D29+/068bXh5hjFP91TenU+hyvAOn++qrXogoooryT2gooooAKKKKACiiszWfE/hzw6EOva7Yaf5nKC4uFjLfQE5P4UDSbdkadFZujeJvDviFWfQtcsdQCffFvcLIV+oByPxrSoBpp2YUUUUCCsjxhpb634T1nR41LSXlhNCgHXcyEL+uK16KBxdndH5+UV6D8XPh3rHhXxdqc1rpN2+kTym5t7lIWMSq/zFNwGAVJIwewB7159Vn1lOopxUo9QooooLCv0f/Zl8AnwJ8HtIsb632Xuqg6reKy8h5QNoI7ERiMEeoNfH/7M/wAHZvir48hn1G2LeH9EZbrUXYfLKc5SH3LEc/7Ib2r9FgABgDAFexllB61X6I+I4sx6fLg4Pzf6L9fuKz6ZpsmfM0+2bPXMSnP6Ui6RpKZ2aZaLnriFR/SrEssUCGWaVI0HVmYAD8awdS8c6FYArFObuQfww8j/AL66flmvUqVYU1ebsfHUqNSo7U43Kvj66isNEFlAqRtdyBSFGMqvJ/XH515vWr4h1+48QXa3EsSxJGuyOMHOBnOc+v8AhWVXzeNrqtVco7dD6zAYd0KKhLfdhXw5/wAFYvHC6P8AB7wv4Dhm23HiPWzduufvQWsZ3DH/AF0nhP4V9x1+Qf8AwU5+JA8Z/tHP4VtJw9n4L0yHTcKcqbiT9/K31xJGh946vL6fPXT7ameaVeTDNd9D5Gooor6U+SPvz/gjT8K/+Eu/aN1j4l3dr5ln4C0R2hk25Ed5d5hj+mYVuvyr9rK+J/8Agkj8Hj8N/wBle28Y39qYtU+IWoS6y5YfOLVP3Nuv0IjeUf8AXb8B9sUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABXxV4r/AOCSX7K3jD4ja38QtTuPGMA1y7kvptHtNThisYppHLyFB5JkVSTkLvwuSBxgL9q0UAfOfgz/AIJ4fsZeBtjaX8BdAvpFIYvrTTapuI7lbl3X6gAD2r3Dwx4G8FeCbb7F4M8H6JoFuBt8rTNPhtUx6bY1AxwK3KKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACvJP2p/2evD37TvwV134V64YoLm6T7VpF+65NhfRgmGXoTtySrAclHcDrXrdFAH8uPjbwZ4l+Hfi7WPAvjHS5dN1vQrySxvrWUcxyoxB56EHGQw4IIIyCKxa/ZH/AIKt/sUy/E7w5J+0f8M9JMvirw7ahPENlbx5fU7FBxOABlpYV692jGP+WaqfxuoAKKKKACiiigAooooAKKKKACv1F/4Je/tEReJ/B1z8A/Et6P7V8NI15ojSNzcWTNl4hnq0TtkD+44xwhr8uq6T4cfEHxN8K/HGjfEHwfem11bRLpbm3c52v2aNwCMo6llYd1YiufFUFWpuHXodWDxLoVVPp19D+hGiuD+B/wAYvC/x3+Gmj/EnwrKog1CLbdWpcNJZXK4EsD/7SnvgblKsOGFd5Xy0ouLcXufZRkpJSjswoooqRhRRRQAUUUUAWLPUL7T5PNsruWBu+xiM/X1roLL4ha3b4W6WC6UdSy7W/McfpXL0VtTr1KfwSsY1cNSq/wASKZ6BD8S7Jl/0jTJkbHRHDD9cVzuveMNS1sNAP9HtT/yyQ8t/vHv/ACrBorWpja9SPLKWhjSy/D05c8Y6hRRTZJYoUMk0ixoOrMcAfjXIdg6iqI1vSWfy0vomb0U5/lV1HWRQ6ng9OMU7BcWiiikAUUUUAYXjrxPH4N8Jan4keMSGyhzGh6NIxCoD7FmGfavibWNZ1PX9Sn1fWL2W6u7hi0kkhyT7D0A7AcCvr7416Ld698NNZs7GNpJ4kS5VFGSwjdXYAeu0GvjSqR7mVxjySl1uXdH1jVNA1GDVtHvZbS7t23RyxnBHsfUHuDwa+0fhz4tHjfwdp3iJkWOadClwi9FlUlWx7EjI9iK+Ia+x/gh4evvDfw40201KNori4L3bRsMFA5yoPvtwSOxOKGGaRj7NSe9zvKKKKk8MKKKKACsPVvA3g3XCW1bwvply5/5aPapv/wC+sZ/WtyigpSad0zzm9/Z++Fl4SyaDLbMephu5R+hYisqX9mT4dyKQl3rURJ6pcpkfnGa9bop3Nliqy+0x/gF4fhr4Yt/CnhSzt7e0gJdpHTdLO5+87t/Ex+mAAAAAAK2Ljxn4kuAQdSZAe0aKv6gZrEorX6xVtyqTscMsPSlN1JRTb6sluLu6u233VzLM3rI5Y/rUVFFZNt6s2SSVkFFFFIDB8feM9I+HfgnXfHevSBNP0DT5tQuOcFljQttH+02MAdyQK/n/APGHinVvHHizWfGWuzebqOuX82o3T9jJK5dsegyxwPSv0s/4KqfG5PD/AIG0f4G6PdML/wATSLqmqhT92yic+Wh/35l3D/rgc9RX5eV7+V0eWm6j6/kfNZxX56qpL7P5hXV/Cf4c638XfiZ4Y+GPh1SdQ8TapBpsThdwiDuA0jD+6i7nPsprlK/ST/gjD8BX8SfEfxH+0HrNlu0/wlAdG0d3Xh7+dMysp9Y4DtI/6eVPavUPHP1u8JeF9G8D+FdG8F+HLUW2laDYQaZYwj/lnBDGsaL+CqK1qKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigBCAwKsAQeCD3r8Z/wDgpn+wBP8ACfWL74//AAb0Mt4I1Scza5pdpF8uhTueZFUdLZ2J4AAjY7eFKgfszUF9Y2Wp2VxpupWcF3aXcTQXFvPGJI5o2BDIynhlIJBB4INAH8r1Ffev/BQj/gnFqfwBnvPi/wDBu0utS+HVxKZL6xAaSfw8WPAY8l7fJwsh5ThXzwzfBVABRRRQAUUUUAFFFFABRRRQB9GfsVftVX/7NnxBMGtSz3HgnxA6Q61arljbsOEuo1/vpn5gPvJkckKR+z+lappuuaZaa1o99Be2F/Alza3MDh45o3AZXVhwQQQQfev51K+1v2Cv22x8IbuD4RfFXUnPgq9lxpt/Ic/2NMzEkMevkMTk/wBw/N0LY8vMMH7Re1p79fM9nLMf7N+xqPR7eX/AP1fopsUsc0aTQyLJHIoZHU5DA8gg9xTq8A+kCiiigAooooAKKKKACiiigBHdY0Z2OAoya5ax0+48S3DanqbuLUMRFEDjI/w9+9dNcRma3lhH8aFfzFNsoVt7OCBVwI41XH4VSdhNXYtvaW1ogjtoEiUdlGKlrd8J6bpWsXM2m6g+yWVN1uwfByMkgDvxzj0BqzqvgXU7El7VluI+3Y/4VvHC1Z0/aQV0c0sZRhU9lN2fnsczRUk9vPbP5dxC8bejDFR1ztNOzOpNNXQUUUUgCvK/Fn7O3gnxJqEmp2U11pE8zF5UttphYnqQhHyn6ED2r1SimaU6s6bvB2PLvCP7PPgjwzfJqd49xrFxEd0a3YXykPY7AOT9SR7V6jRRQFSrOo7zdwooopGYUUUUAFFFFABRRRQAUUUUAFFFFABWX4q8TaL4L8Nap4u8R3qWml6NaSXt5O3RI41LMfc4HA7nArUr83v+CoP7TCXEkX7OPg+/BSFo73xRNE3Bbh4bTPt8sr+/ljPDCt8NQdaooI58ViI0KTqP5ep8U/Hf4ua18cvit4g+JWtGRP7Uuj9jt3bP2W1X5YYh2+VAASOrbj3rgaKK+rjFRSiuh8ZKTlJyluy1pOlalr2q2Wh6NZS3moajcR2lrbxLueaV2CoijuSxAA96/pB/ZQ+BNh+zh8BPCnwpt1ha+0+1+0avcRDi4vpTvnfP8QDkqpP8CIO1fh5+wl8RPgR8Ifj1p3xR+PMuqtY+HIXudHgsbAXKtfEYSSQbgQsYLMuATv2HgKa/Xnw3/wAFPP2JPEhWJPjMmmzt/wAs9S0e+t8dOrmHy+/97se1Mk+p6K8u8L/tSfs2eNNq+GPj14Bv5XGRAniC1WbpnmJnDj8R6+lem29xb3cKXNrPHNFIMpJGwZWHqCODQBJRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAR3Fvb3lvLaXcEc8E6GOWKRQyOpGCpB4IIOCDX5Kft5f8Esb3w8+ofGD9l/Q5bzSiWudV8IWwLzWfGWks15MkfUmEZZc/ICvyp+t9FAH8q5BUlWBBBwQe1FfuH+21/wTI8C/tCfb/iN8KPsXhL4iSbp7gbNmn6055Pnqo/dyk8+coOSTvVidy/jL8S/hd8QPg74uvfAnxM8KX/h/XLBsS2t2mNwzw6OMrIhxw6EqexNAHLUUUUAFFFFABRRRQAUUUUAfcn7D37e138NprD4SfGfVZLjwgQtvpWrS5eTSDn5Y5D1aDnAPJj4/h4X9TLS7tNQtIb+wuobm2uY1mhmhcPHKjDKsrDgggggjrX859fWP7Hf7dniX4AXFv4H8dm713wBK+FiU77nSSerwZPzR9zESB1K4OQ3k43Ac/wC8pb9j2svzPktSrPTo+3/AP2EorF8G+NPCnxC8N2Xi/wAE69Z6zo2ox+ZbXlrJuRx0IPdWByCpAKkEEAgitqvDaadmfRppq6CiiikAUUUUAFFFFABRRRQBQ1xtag02S+8NOi6xYkXdhvOFeZDuWNj1CPgo2P4XYV6v8MPiP4e+LHguw8aeHJD5F2pSe3k/1tpOvEkMg7Op4PrwRwRXm9fOvizxr4s/ZL+LkvxF8NWDah4K8aShtZ0vdtQXQyWZGwdjnJdT/Fl1IwAR6OX4r2UuSWzOPGZd9bhaHxrbz8v8v+Cfd95o9leIUkhXB7FQV/I1zeoeArSQlrdGjPrG2R+R/pU3w0+KXgj4ueGovFXgXWor+0fCyp92a2fGTHKnVGHvweoJGDXWV7k6VKqrySZ8tGpXw8nFNxa6f8A8uuvBeoQk+TNG/s4KNWbNoGsQZ3WEjf7mG/lXsTKrjDKCPcVA9jaP1hA+nFcc8rov4bo7aecYiPxWZ4xJbXEP+tgkT/eUio69mbSrY9GcfjUZ0W2JzuP4qKweUdp/h/wTpWd94fj/AMA8dqVLW5k/1dvK3+6hNevLo0C9HYZ9ABT10q2HUufqaFlC6z/D/gg87fSH4/8AAPJ4tE1eX7mnzD/eXb/Ortv4Q1qc48qND7vn+Wa9RSxtU6QqfrzXG+Kfi34U8N6yPCVhKNW8RFQ7abZsCbZD0kuHHywp0xu+Zv4VbpVvLcPTXNUkyI5piq0uSjFXKcfw/miia51LUo7eJBudtvAH1JrmtQWwS6ZNOeV4F4DyYy59cdhVrWPEmq66V+3SqqKBiKIEID64zz+NZleZiJ0X7tGNl36nr4anXXvV5XfbogooorlOsKKKKACiiigAoorifjF8XvBfwO8Baj8QvHN+ILGxXbFChXzryYg7IYlJG52x07AEnABIqMXJ2QpSUU5S2PPf2wv2mtK/Zs+GM2q20sE/izWVe10CyfDZlx807r/zzjyCfUlV43ZH4m6vq2p6/qt5rmtX897qGoTvdXVzO5eSaV2LO7MeSSSST7123x1+NvjD4/8AxG1D4h+MZ8S3J8qzs0cmGwtlJ2Qx57DJJOBuYsx5Jrz6vpcHhVQhru9z5HH4x4ipp8K2Cur+FXwx8XfGX4iaD8MPAunm81vxDdraWyHOxM8vI5AO1EQM7NjhVJ7Vyle6/softVeJP2QvG+qeMPDnw98Na9qeoWy6fJJrMdwLiziD7pEhZJFEZchdxZGPyKBgbg3YcJ+oI/4I0/sqz6HY2F3rfjqHUYLWOK6v7LVIUNxKoG+Ty5IZFXcQTtHABwOxrg/EP/BEH4a3KuPCfx28Tacx+6dR0q3vQOO4Roc849KzvBX/AAXA8K3HlxfEb4DarY4yJLjRNYju93uIpUix9N5+vavoLwH/AMFV/wBjLxt5UV7491Pwrczfdg17SJo8H0MsIkiX8XxQB8a+KP8AgiN8YrMMfBfxm8Hatj7o1O1urAnn/YWfHGf8815ld/8ABN//AIKDfCOWS/8AAmj3MoTLPceFfFUcDntwpkikboOik4/HH7T+Bvi78KvidD9o+HPxI8MeJk27j/ZOqwXTKP8AaVGJX8QK62gD8Jm+O3/BU34BOG8Ral8WLS3gbDS+ItFfU7eQDgjz7mKQMPdXz3z0Ndt4F/4LR/tF6EEt/HHgfwZ4ohU/NLHDNYXL/V0do/yjr9o64fxt8DPgv8SvMb4gfCfwj4iklzul1LRre4lz6iRkLA+4OaAPh3wJ/wAFsfgpq7xw/EP4UeLfDbyYBk0+eDU4Yz6sSYXx7hCfavpL4ef8FB/2O/iWY4dF+OWg6dcv/wAu+ul9KYH+7uuVRGP+6xzXCePP+CUH7GfjUyS6d4M1nwlcS8tNoOsSrz7Rz+bGv0CAe1fNXxB/4IgzgSXPwq+OyOc/u7LxDpRX87iFj/6JoA/UzStY0nXbGPU9E1S01Gzm5juLWdZon+jKSDVyvxQ8Gf8ABNv/AIKF/CD4i6VJ8OtRttLQ30Rm8QaF4ojitYFDj55YXaOWVAOSnlNuGQRziv2voAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAK8v+Pv7Nfwf/aW8Jnwn8V/CsV+sYJstQhxFf6e5/jgnAyvYlTlGwNysOK9QooA/Br9rb/gmh8Z/wBm9rzxZ4Win8deA4cynVbG3P2qwTk/6VbgkqFAOZVynGTsyFr48r+qivin9qj/AIJafBL48G78VfD5Yvh54ylzI1xYW4OnXr9T59qMBWP/AD0jKnklg/SgD8LaK9k/aE/ZG+O/7MurNZ/FDwZPFprvstdcss3Gm3XJA2zgYVjj7kgV++3GDXjdABRRRQAUUUUAFFFFAHrf7PX7TvxP/Zu8R/2r4K1IXGlXUgbUtEumLWd6OmSo+5IB0kXkYAORlT+t/wCzn+1t8Kf2kdKX/hGNSGm+I4YvMvvD97IBdQ46snQTR5/jXpkbgpOK/DWrek6vqug6nbazoepXWn6hZyCa3urWZopYXHRldSCpHqDXFisFCtrs+534TMKmH93ePb/I/oqor8zf2b/+ComsaMLXwp+0PYy6rZjbFH4jsIh9qiHTM8IwJQO7phsD7rk1+iXgb4geCfiX4fg8U+AfFGna7pVwBtubKYOFOM7WHVGGeVYBh0IFeDXw1Si7TXzPpcPi6VdXg9e3U6Ciiiuc6QooooAKKKKACsPxt4N0Tx/4YvvCniG382zvo9pI+9Gw5V1PZlOCP8M1uUUxptO6Pgm88F/tAfsveI5vGXhltUsrG3l8ldasV82zuI8/Ks68qAf7ko6jjOAa+kfhX/wUi0G8hh034v8AhabTroAK2p6Qplgc+rQsd6f8BL/QV7lpeq3WkzmWDa6SDZLE4ykq91YVgeJ/2Vv2cfjGkl83hNdA1eRSZZdFkFpIGPVvKAMTcnO4oSe9etg6smrU5WfZ9fQ58fWw1XXHUrr+aO69f6a8j0fwd8fPgx4+RD4U+JWg3kkn3bd7oQXB/wC2Mm2T/wAdrvq+DfFn/BMu+V3l8DfFGCRCflt9WsWQqPeWMnP/AHwK46P9j79srwFmLwX4mcpH93+xPE8lqpwD0DmLHp2616Ht60fjh9x5Ly3LqutDEpeUlb8dPyP0jor8518J/wDBSDSs2ou/Fr/xZOtWlx/4/wCa3p0zUNtpn/BRzUUZIpPGaA/IfMu7aA8+hZgfxHSn9af8j+4n+w47rEU//Aj9Hq8/+IHx9+D3wwikPjPx9pVpcRg/6FFN590T6eTHucemSAPevjK1/ZT/AG1viN+6+IPxEu9OtGP7yLVvE010AOM7YoWkTPtkD3ryP9oj9m2X4F+JfDHg2x8SP4k1bX7YzMsdr5ADmXy0RF3MTk5GSeSOgqKmKqRjzKFvU6MJkuDqVlSliFJvpFfrse9+KP2t/il+0J4pPw4+ANrP4W0dwTe65OAbxYOhfIJEI54CkuTjDLzXq3w/8AaF8OtCXRdGWSWSRjNeXs7b7i8mP3pZH6kn9OgrI+DPwr0v4TeDLbQbVEk1CZVn1O6HJnnI5wf7q9FHpz1JrvK8KviJ1ZXkz2VTo0l7LDxtH8X5thRRRXOSFFFFABRRRQAUUVxvxZ+L3gD4JeD7nxx8RNdi03ToPkjX7011KRxFDH1dzg8DoAScAEhxi5OyFKSiuaTsi98RfiL4P+FPg/UfHfjvWYdM0fTI980z8sx/hRF6u7HgKOSTX4t/tVftQ+LP2mfHb6xfNcaf4Z052j0PRjJlbaPoZHxw0r4yx5xwoOAKd+1J+1Z45/ab8VLeatnTPDWmyN/ZGixOTHCCSPMkP/LSYrwW6DooAznxCvocFglSXPP4vyPl8wzB137On8P5hRRX2h/wT4/YD1r9pvxLD4/+I2nXunfC/Spd0svzRPrkyn/j3gbr5YIIkkXpgop3ElPRPLO//wCCXn7CD/FfXrX9oX4taM3/AAhei3G/QdPuY/l1q7Q/61gesETD6O428hHB/XPxZ8Mvht4+QxeOvh74a8RowwV1bSbe8BGMYxIjdq2tG0fSvD2kWWgaFp1vYabptulpaWtvGEigiRQqIijgKFAAA7CrlAHzL42/4Jt/sX+ORI938FNO0m4cYWfRLq40/wAv3EcTiL80NfPPjz/giX8HtUEkvw4+Lvivw/KwJWPVLaDUolPYDYIWA+rE1+kFFAH4n+O/+COn7U3gyf8AtL4eeIvC3i1YW3wC0v30+9DDkHbMFjU9MYlPPp1rjD8Rv+Cnf7JpJ1rUPihpOm2h5bWbZtY0tVHVVlmWaFR/uMOuRjrX7x0UAfj38Mf+C2HxV0fybT4tfCnw/wCJIV+V7vSLiTTrgj+8VbzY2PsAg+lfXPwt/wCCsf7InxEMVpr3iLV/A1/IAvleILAiEt3AngMiAf7TlPw6V7Z8T/2Q/wBmb4xiWT4hfBXwxqF3P/rL+C0Fnet9bmApKf8AvrufWvkP4q/8EVvg9ryz3vwj+JPiDwndNlktNTjTU7MHsoP7uVR2yXcjrg9KAPv7wf478E/ELSV17wH4v0XxHprYAu9Kv4ruLJGcb42IB9q3a/DLxj/wTR/bk/Z/1R/FPwy8zXDagtHqngnWZIb1Fzx+6PlT7uBkRh/qaf4K/wCCm37b/wAA9V/4Rf4oIdfNqQkum+M9Hkt76JR1/eqIpd3vJv8ApQB+5VFfLf7Ef7d3hr9sm11+xg8EXvhbX/DUUE15ayXa3VvNHKXUPFKFU8NGcqyDG5cFucfUlABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFVtSuZLPT7m7iTe8MTOq+pAzXl02va5eS/PqV0Wc8KkhUZ9gK8zMM0p4NqMottnTh8LKtdp2setUVgeHNO1uyhFzrGrSOGXJgkO7b9WPOamu/F3h+0Yo9+sjDtEC/6jj9a6I4uCpqpW9y/RtGbpPm5Ye96GzRXNf8ACf6F/duv+/Y/xqeDxt4emIDXTxE/89Iz/MZqY5jhG7KovvG8PVX2Wb1FVY9U02aMSxX0DqehVwf0pG1GIfcilf327R+ZxXbFcy5o6owk1F2Zboqg2oTH7sUa/wC85P8AIf1qJry5b/luF/3YwP1JNaKmyPaI1KKyDPK33ppj/wADx/ICmFgeoJ/3nY/zNP2QvamySByTik8xP76/nWNiPOfJi/GMH+lITEo3NFAB7xr/AIU/ZeYva+Rt5zyKWuW1HUbS1t/NAETg5Vk+Qt7Ad62dBvX1DSbe7kOWkDZ/BiP6Up0nFXHCqpS5TQooorI1Kmq6TpWvabc6NrmmWmo2F5GYri1u4VmhmQ9VdGBDA+hFfAX7Rv8AwR7+EPxCe58R/ArWW+H2tSEyHTZVa50iZvQLnzLfJ7qWUDgRiv0IooA/m6+PX7In7QH7Nt26/FHwBeW2l+b5UOt2Y+06bOf4cTrwpPZX2t/s145X9UV5Z2eo2k1hqFpDdWtwhimhmjDxyIRgqynggjgg18a/Hz/glJ+zN8YDc6t4N0+b4ba/MS4uNCjU2LNz9+yYhAOekRi6daAPwnor6++On/BLf9qf4OPPqGieGY/iBocYLi98NBprhV9HtCBMGwMnYHUf3q+Rru0urC6lsr62ltriBzHLDKhR42BwVZTyCD2NAEVFFFABRRRQAV1nw3+K/wARvhDrw8SfDbxfqOg3/Aka2k/dzqOQssZyki8/dcEVydFJpNWY4ycXeL1P0n+Bv/BVnTbr7NoXx98KGykwEOvaKheIn+9LanLL6lo2bJPCAV9z+APif8Pfipoy6/8ADvxjpev2JxuksrgO0RPRZE+9G3+ywB9q/nxrV8MeLPFHgrV4tf8AB/iLUtE1KD/V3en3T28q+25CDj2rza2WU5609H+B62HzerDSquZfif0QUV+TXwk/4Kj/ABr8G+Tp3xK0nTvHGnphTM+LK/A/66opRuP70eT/AHq+yfhf/wAFFf2Z/iP5NrqPiifwdqMmAbfxBCIIs98XCloQPdmU+3XHl1cDXp7q68j2KOY4ertKz89D6coqrperaVrljFqei6naahZzjdFcWsyyxOPUMpIP4VarkO4KKKKQBT4ppbeRZoJGjkQ5VlOCPxplFMGr7nYaR8RLu2VYdWt/tKKMeYnEn454P6V1ll4s8P3wXy9SjjZjjZKdhz6c8V5HRXfRzKtBWevqeZXyqhUd4+6/I9xR0kUOjBlPQg5FOrw1WZTlWI+hp8lxcTf66eR+CvzMTweo/Sur+11/J+P/AADjeSO+k/w/4J7HeaxpdgD9s1CCIjqrON35da+U9Y8Ban8Qfj5c/GjxkiQ2ekwrp/hvTCwd4403fvpSPlBLPI6qCcbxkgrivRao3ms2Fl8jzeZL2ii+Zz+A/rXLicfOsuRKyPRwOBjhJOad21YszXCQvEjfemfYo/An+QqWsnToL+8vP7W1KPyQqlbeA9UB6k+9a1cDO5BRRRSAKKKbLLHDG800ixxxqWd2OAoHJJPYUAOor5t+M/7f/wCzx8IFnsLfxL/wl+uRAhdO0FlnVW6YkuM+UnIwQGZh/dNfn18e/wDgoN8cvjSt1omkX48F+GZwYzp2kysJ5kPaa54d8gkELsUg4Kmu2hga1XW1l5nBiMxoUdL3fZH3v+0t+318KPgRHd+HdAnh8X+Mosxf2bZTD7PZycj/AEmYZCkEcxrl+xC53D8pvjJ8cPiT8efFb+LviRr731wNy2ttGClrZRk58uGLOEXgc8s2AWLHmuDor3MNg6dFXWr7nz2Kx1XEO0tF2Citbwp4S8T+OvENj4T8G6Bf63rOpS+TaWNjA008zdcKigk4AJJ7AEngV+tf7FX/AASd0XwO9h8TP2nLWz1rX02z2fhZWWaxsW6g3LDK3Eg/uDMY5yZOMdRxHzn+wX/wTQ8RfHefT/it8bLG80T4dcXFnZEmG814dtv8UVue8nDOPucHev7S6HoejeGdGsfDvh3S7XTdL02BLWzs7WIRwwRIAqoijhQAAABVxESNFjjUKqjCqBgAelOoAKKKKACiiigAooooAKKKKACud8cfDrwB8TNIbQPiJ4L0TxLpzA/6NqthFdIpPUqHB2npyMEYHNdFRQB518If2efgt8BI9Vj+EPw903wyNbkSW/NqZGacpnYCzsxCrubCghRuOBzXotFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAViaz4t0zRpTbPvmuAMmOP+H6ntW3XEa74K1K91aW7s5YmiuG3nexBQ9/qK8/MquJp0r4WN3f7jow0acp/vXZGronim6124MUOilYF4klM+Qo+m3k+1aMejaHp0zX6WVvA458w8BfpngfhT9F0qHRtPjsojuK/M74xvY9TWF4w8P6vrN1bvZMrQom0oz7QrZPzflj8qykq9LDKpWj7Soullp6WXQpckqjjB8sRfE9tdeI0ji0PU7W4jiBMsEc4yT2PHH51wWo2N/ZM9rPG1vPt43r09/cV6vomkQaLYJaRBS+MyOBgu3rXMeJ7e0vopJ5Z5IJY3y25N4HYgEHOPwrix2USxFKWKk7TSu109PL8jWjjVSmqS1i38zzjUJpUa2tHSSQsVLPH8uT3/wAatX96lhCJnRmBbbx/n2rUFnaEZOrQD6xyZ/8AQab9mtMHdqKH0CxMc/nivlfYy8vvX+Z63OvP7mQ2l3PbslzayyQvjIKnB+ldTpHixZSLfU8I54Eo4B+vp9a5zytOHW9uDj0thg/jv/pTZJdKgRpJFuiqjJYyqg/Laf5124LGV8HPmpzVu17p/dcwr0adZWlH5npIIIBByD0NIzKvLMB9TXnWm+L0hc2dlDtIGRHO0hOPYZA/Ste48Yv9jkEVjHDNsPzrzz7d/wBa+tpcQ4OS/eNp+j/B/wCdjxp5bXT93VHWeahGVywHdQSKp3Otafaf6+7gj/3pAT+Qya8zuNfvNQOZ9Rkf2LEVXznnNZ1c/h/y5hf1f+RMcDL7bO/ufGmlpkRzSSEf3U2j8zn+VZ8vi+KZvlk8r32En8zn9MVyFFcjz3E30S+5/wCZp9Sp+Z013cJdRCdJvMy3LZz2713ngx92gQjP3HcfT5if615JY3AW6EAb74OR9K9U8CPu0Vlz9ydh+in+te/hcX9awvtGrNOzOKVL2Vfl7o6OiiiqNgooooAKKKKACvMPjD+zJ8BPj3btF8WPhdomu3BXYt+0Jhvox2C3URWVR04DY4HFen0UAfmB8Zf+CJ/hy+83UvgJ8VbnS5TkrpfiaP7RASey3UKh0UehjkPv6/DXxe/YF/aw+CxmuPE/wj1XUtMhyx1TQV/tK12j+NjDl4x/10VP1Ff0SUUAfysOjxu0cilWU4ZSMEH0pK/pV+Kv7Lf7PPxuEj/FD4ReHNbupQQ181r5F7j0F1EVmH4PXxx8Uv8Agit8G9f868+E3xK8Q+ErhsslrqMSapaA9lXmOVR0GS7kdcHpQB+ONFfanxO/4JH/ALW/gTzrrwzpWheObKP5g+i6isc+33huBGSfZC/418reO/hL8Ufhdd/YviP8OvEnhiYttUatpk1qHP8Ass6gMPcEg0AcpRRRQAUUUUAdF4M+I/xA+HN9/aXgLxrrfh64JBd9NvpLfzMdmCkBh7HIr6U+H/8AwU5/aU8ICO28SXOh+MbVcKf7TsRFOFHpLAU593V/618k0VlUo06nxxTNqWIq0/4cmj9RPA//AAVp+GmpCOH4hfDPX9ClbhpdNuIr+EH1O7ymA9gGI9+te9eD/wBuX9lbxqEWw+MGkafK3DRawsmnbD6F51VPxDEe9fiDRXHPLKEvhujvp5xiI/FZ/wBeR/RFoXifw14otvtnhnxDpmr24/5a2N3HcJ/30hIrTr+dCyvr3TrlLzT7ye1uIzlJYZCjqfZhyK9J8N/tP/tFeEgiaF8a/GMMUeAkMurTXEKgZ6RyMyjr6Vyyyl/ZkdcM7j9uH4n7zUV+Mmhf8FHf2tNGAS58f2WrIvAW/wBGtD/49HGjH8Sa7zS/+CsH7QNqvl6p4N8CXwAOHFndRPnPci4Ix+ArCWV11tZnTHOMO97r5H6v0V+ZFp/wV08cJITf/BrQpo9vCw6pNGQfXJRuOvGKt/8AD3nxH/0Q3Tf/AAeyf/GKj+zsR/L+KNP7Vwv834M/Sw88GmqiJ9xFX6DFfmLef8FcviE5l/s/4QeHYQQfK86/nl2nHfAXdz6Y/rXN6p/wVg/aDugU0zwf4DsVIHzGyu5XB78m424/4DTWW4h9PxJebYZdfwP1gor8ZNd/4KO/taayClt4/stJRuCtjo1oP/HpI3YfgRXl/ib9pb9oLxiHTxF8Z/GNzFJ96BdXmihP/bJGCfpW0cqqv4mkYyzqivhi2fuX4r+IXgLwHALrxv420Hw/ERkPqeow2ob6eYwz+FeCeO/+CjP7LPgkSRWnjG98T3UfWDQ7B5c/SWTZEfwc1+M800txK088rySOdzO7Esx9ST1ptdMMqpr45N/gclTOqr+CKX4/5H6AfEj/AIK1eLr9ZbP4U/DLTtIU5Vb7WrlruUj1EMexUP1ZxXyR8UP2kPjj8ZGdPiJ8SdY1O0c5+wLKLezHp/o8QWPI9SufevNqK7qWFpU/giedWxler8cgoqaysb3U7uHT9Os57u6uHEcMEEZkkkY9FVRySfQV9k/AD/glN+0v8YXt9V8Z6ZH8NvD8u12uddjb7c6nGfLsgRIGwekpiHvW5zHxhX2H+y9/wTF+Pn7QRs/Eniazf4f+DJsSf2lq1u32u6j/AOne0OHbPBDuUQg5BbpX6jfs3/8ABOv9nD9nE22tad4bPirxXAQ41/X1S4licd4IseXBg5wyrvwcFzX0/QB4z+zj+yP8Ef2XNCOmfDHwwo1K4jEd/rt9tm1K96ZDzYG1cgHy0CpkZ255r2aiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAK5bxBaA3E8R+7OmR+Ix/OuprH8Qw5jinA+6Sp/H/APVWtKzfLLZ6GVa/LzLoeWkFSVYYIODUUFxDcoZIXDKCVz7itDVofI1CZQMBjvH481SCqowoA+lfluIoujVlSlvFtfcfVUpqcFNdULUF7bfa7WS33bd44Pv1p88bTQvEkhRmGAw7VHb200FqkH2ks6nJcjORnOPy4rIsz7DSr2O+F5ezBigwMMSTxitggEEHoaKktNHvbqd7m0s7qYuAp2IWUY+gqkpTdkrsWiWpyTqUdkPVTigMy/dYj6GtPVtD1a0u5vO0y5QbtxJiOBnn+tc/c2U015FcLOUWMglcc8Z/nmupwlH41Y57p7alubUfsqgzXJQHpk5zS/bGdVb7TlX+6d3WqGpWE90Ve3nEbbSjA9Cpqhc6ReqiQQBJl2AFjgFTknjPbmmteomvI6nSWxqMJJ6kj9DXsPgB86dcx5+7Pn81H+FeMaSGhuLVHfcylVLep6V7D8PXzFfR5PDIfz3f4V9fw+74OpHtL9EeNj1bExfl/mdfRRRXrGQUUUUAFFFFABRRRQAUUUUAFFFFABUF7Y2WpWkthqNnBdW067JYZ4w8bj0Kngj61PXwt/wUy/bok/Z38LJ8JfhhqgT4i+JbYyS3cRBbQ7Jsr52e00mCIx/CAznGE3AHzL/wU6+I/wCxpoF7qPwp+FfwS8Hah8Q1kMer6/plubKHR3HVB9mZFnuf7wbcqEYYM2VX82qdNNLcSvPPK8ksjF3d2JZmPJJJ6mm0AFFFFABRRRQAUV9BfBn4SeBfix8BviDc2ukyf8LA8Jr/AGhaTpdSfv7baH2eTnax/dTLkDq6e2fn2gAooooAKKKKACiiu60j4N+Lta+Eet/Gmzl08eH9Av1067V5mFwZGMAG1NuCM3MfO4d/TkA4WiiigAooooAKdHHJNIsUSM7uwVVUZLE9AB3Ne8fsXeIP2d9I+NGnaZ+0x4AsvEHhTWWWzS+uby5hXR5ycJO6RSKkkRztcOG2jDjG0hv31+H/AMEPgt8MUjk+Gvws8JeHW28XGl6TBBK4I6mVV3NkdyTQB+A/ws/YY/au+MTRS+EPgvr8NjKRjUNXiGm2u3+8r3BTeP8Ac3H2r7j+Cn/BFC1iNvqv7QPxTM7DDSaP4XjKp9Gu5lyR2IWId8N3r9TqKAPLPgz+y98A/wBn+1EPwn+GWj6Lc7Nkmo+WZ7+UY5DXMhaUg8/Lu28nAFep0UUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFVdSh8+xlQdQu4fhzVqvPfjf8e/hV+zv4Mfxz8WvFMOkacX8i3j2GW4vJcZEUMS5Z2wOwwByxABNNOzuJq6sUPEkOHhuB3BQ/wAx/WsKfzTC/kEeZtO3PrX56fE//gsa95rUln8Mfg6j6FGzeXca7elLuY4O1vLiykeCRldz5HcZ48svf+Csvx5luFax8FeC4IBISySQXMjFM9AwmXBx3wfpXzGb5RWr4yVaglZ2+/qejgcXGnQUKm6P1XsBei3P23HmFiVGQSB2BIrJsG1O51W2t4Ybgzs+yVckh8n06DivgX4Rf8FVP7d16z0L4t+FYdDtruVYjrGnTl4Lck4BkhcFgnqwYkD+E19SftVfHDXfgd8B77X9MvvI8TeLJv7E0GaMgSWysha4ulK9CkeFUjkPIp7V5NPKcR9YjQqK1/y6nbLF0/ZucXsYf7QX7ZkPgDxT/wAKb+A3huPx38R5Zvsc0iQtdWljP3hjiTm4nXnP8CEYO4hlHPaV+xt+3R8c4U1345/tE3vhWK7O/wDsqK7knaEHrm1t3jtk47K3bBxXO6NY67/wTr+AnhD4w2/gXSdd+IHxDu/L1a41hJCdKt3iM0drGUYFWIG5znlgQchFwniL/gsD42uNFtYvCXwc0Sx1fZ/pVxqOozXdsWz/AARIImxjB5fvjtk/aYbC0sPDkpK39dTxqlWdR3mxnjH/AIJh/tFeBrv+2/g1+15qEWqeWHWKRb3SUdhwQZIbiY8hV6rg45rjNH/bA/aO/Zm8ZWfw2/bf8CXGpaVdsUtfFFpBGLoIDjzUlixDdooI3JhZQDknOFPkf7Q3/BR39p/4k+Dh4Wk17S/DlpeyFLq48P20lpdSJj/VecZGZUPfbgnGM4JB9y/4J3Q/DH9pn9lbxT+zF8U7uHVb7SdUuL6ztJF/0qwtJlQx3VvIwOHW4eflc43gMMSYO1SlCpHkmrozUpRfMj7D0nVtF8RaLp/ifwxrFrq+i6vbrdafqFq+6K4iPQg9iOhU8qQQQCKdbzSytIJLdowjYUk/eHrXxV+xRr3i/wCAXx78afsNfEbUGuoI5Z7/AMMTuCF89E8790CfljuLbMu3OFdPVmNfbdfFZlgvq1W0fhe3+R7GGre0jd7oltm23MTejqf1r1v4fPi4vUyeUQ/kT/jXkCnawb0Oa9Z8AP8A8TS4TJw1uW9uGX/Gvb4blejWj6fqefmStVpv1O8ooor3DnCiiigAooooAKKKKACiiigAooooA4H48fGPwz8AfhJ4l+LXixwbLw/ZtMkAfa93OcLDApwcNJIyIDjA3ZPANfzefE/4keLPi/4/134leONQN7rfiC8e8upOiqT91EH8KKoCqvZVA7V+jf8AwWp+O013rnhH9nXRrv8A0awi/wCEm1tUIIaZ98VrGe4Kp5zkHg+bGe1fl7QAUUUUAFFFFABRRRQB7V+x/wDE63+GXxy0O41fUo7HQ9ekGi6rPMT5UMMzKFlb2RwjE/3Q3rVb9rL4I3/wD+N+v+CZbUw6fJM17phC4U27scKPZCGT3Cg9xXj1feCRp+3P+yVEIdtz8X/g3AlvKp5n1bTsBYpMn7xKqI26nzI42JAen5CPg+ildHjdo5FKspwykYIPpSUhhRRRQAV9h/syaBaeO/2Lf2gPDlsXfVNBa01kQgj54mVmyARyQ9mo46bh3Ir48r6e/wCCeXxV0n4f/HuPwl4ruEi8OfEfT5PCl80vMUUszKbaRgeOJlRCTwFkYmgGfMNFeyftFfsyfEn4DeLNdTV/B2sQ+FbbVnsdO1mS1f7LOCN6IJSNrMFODz95W9K8boAKKKKACv2x/wCCTH7Vc/xg+FVx8FfGWpmfxV8P4EFlJK+ZLzSiQkZyephbERP91oepJr8Tq9j/AGQ/jjc/s7/tDeDvicJ3TTbS9FprKKTiSwm/dzggfeKq29Qf4kU9qAP6RaKbHJHNGssUiujgMrKchgehB7inUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABX4kf8Fk/EPibUP2p9P8AD+qzzDSdI8MWr6VASRGBLJK0sgGcbmddpYdREo/hr9t6+N/+Chn7B0v7W+jaR4r8DatYaV488Nwta27XxZbbUbVm3+RI6qzIysWZGwRl3BHzblAPwfor7S03/gkZ+2Hd3UkWqaT4V0e2jk2fa7vXEeNhnG4CFXfH1UHnpXs/gP8A4JEaJoV1FP8AFr4jzau4CyGy0eL7NCfVWlcM7r16BDWGJxEMPS9tV22LpQdSfs47n57/AAo+FPi74x+MLTwf4QsHlmmYNc3JU+TZw5+aWRuyj06k4AySBX6O/tE2UfiD4w/sxfCHU5JLvRIp7S03THd5iXGpR28mR3+S3UE98Adq+rfhv8CPhr8KdGTQPBPhmy0qxQ7jHbxndI3955GJd292JNeA/wDBQfw7qvhPS/hP8cfDEG9/A+tSWUxwSI2MqXdqTjou6OZc+pUV5GCzSOLxnKlZJO34HZWwrpUb+Z0X/BYLXtdtfBPw58NWrONH1HU728vcH5TNDHEsIP8AwGeY/h9K/L2v2t/af8GfDT9rn9mvRbq18caFoc2tm31fwlqWqXaQRNdshxAxJzllZ0ZVBZWGdpKYr8r/AIlfslftEfCS11DVPHHwv1W00rTPmuNTh2XFmqbgoYzRkqASwxnB5AxniveOA8F8aQebojSY/wBTIr/+y/8As1e6/wDBLS71S2/bF8Nw6eX8i70zUYb7axA8kWzuMjHI8xIuOOcHPGD5Za+Hf+EtuoPDP9rafph1OVbVbzUJGjtoGYgB5XVWKoDjLYOBzX6x/sS/sI6B+ynDe+Ktc1yLxF451e2+yT3sMRjtrKAlWaGAN8zbmVS0jAE7VAVedzQpPQ8H/bkt4/Bv/BQj9n3x7pYeO51O50y0u1hT55UTUTG31Zopyn0UCvsu9jWG8nhXGI5WUfgTXxD4/wBdt/2mP+CpXhjTNAmFzoHwjEct3dRcxq9g73MrZH/T08cHplR65P2ssaJJLIgIM0hlbnPJ6185xDKNqa66/od+AT1Y6vUfAEmdWQ5/1luf6H+leUwwGJ5H8138xt2GPC/SvTPAEn/E0sOvzwkf+OH/AAp8Nv3qsfJGeaL+G/M9Pooor6E5QooooAKKKKACiiigAooooAKKK5X4r+JJfB3wt8Y+L4JCkmh6BfakjA42mG3eQHqP7vqKAP53/wBrv4nTfGH9pf4jeP3uTPb32vT29i5P/LpAfIt//IUSfjmvIaKKACiiigAooooAKKKKACvQfgP8afFXwA+J2k/EvwmyyS2TGG9spDiHULR+JreQd1dfY4IVhyorz6igD7E/bD+Bfhbxb4ds/wBrr4AZv/BHizM+r2kYBm0m7/5arKoztZWOHHqQ4LK5K/Hde+/smftR337P3iK70HxNYNr3w48VFbfxLobgOCh+X7TCrfKJkB6Hh1+Rv4SvVftY/sk2/gK0g+OHwRvo/E/wm8Sp9usr6y+YWAZuY3H3gqsduCNyEFXwQCwI+WKKK+jf2V/2FvjJ+1HfRalpFkfD3g1JNt14k1CFvJbBwy28fBuHGDwpCgjDOpIyDPBvC/hXxL431+y8K+D9BvtZ1jUZPKtbGxgaaaZuuFVQScAEk9gCTwK/Tn9mP/gl94O+HFhZ/FX9r7V9OaWGWJoPDn2pVsLWRmAQXc4OJW3MB5anyxj5mkUkD6c+GPwo/Zq/Yd8LSaZ4QsUu/FF3p7TyXtxibVdYOGKR+aF2xq7RlUjXapZeAzZJrNa/ED4/ajBrFz4g0mPwPdR/Z9Utmlkl07GMmOJZEiaSVQY3E38EoYZHl7WwqV1F8sNZHs4DJ51o/WcU/Z0VvJ9fKPdvp0voe2fGD4U+Efjj8M9Z+G3iy2juNK1u1MauoBML9Y5UPZlbDA+1fz2fG34QeKvgT8TNb+GPjC3ZL7SJysc20hLqE8xzJ6qy4PscjqDX7/fEP4j+GP2bvhO/iDxj4hv9WGmxGC1N7MjXup3ByUiBVQCx9dvyqCT0Jr8e/jr4i8S/tP6vqeueKpoZPFEkrXOjFF2oq4x9gT0Rgo2ZziQf9NXat1qtTx5JKTUXddz5MopWVlYqykMDgg9QaSgQUUUUAf0R/sAfE+T4tfsifDnxJeTmW/sdM/sS9LHLmWzdrYMx7lkiR/8AgfrX0LX56f8ABFTxNNqP7O3i7wvPKX/sbxbJNECPuRz20B2/TfHIfX5j7V+hdABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUARXUXn28kP8AfUgfWvN/EsPyw3GOhKH+Y/ka9NrifFFn8l3EB9394v8AOufHUfb4KrS8rr1WoU5+zxEKnnb7zhri4jtojNJnAwOO5PFY/iTw14V+KvgbXPh/4ut3l0PxDbmzuwoHmQOCGjmTqN8bhXX3FbUsUc8bRSoGRuCD3qNEtLCEIuyGPPGTjn6mvzqjWlSmqkN0fSzgppxlsz4I0PxJq/7IN7qv7Ln7VPhE+MfhD4kaSfS9Rt4RJ9nD8fa7Nm6dcvECHjf5l5J8zS+KPw7+LfjD9nDxAfCH7Zei/Ej4XeE7BNQttGhiA1N4InUpHdLt81fLADBZXONgwq8Y+3PEeheFfG+g3Hgzxz4d0zxFos5EkunX8YkVW5AkQ/ejcZOHUgjPWvl7xn/wTR+COv3Ul54G8e+J/B5l5+zXNvHqtvF14Q7opMf7zMevPp9phc7w1WK9o+WXnt9541XBVIP3VdHwRfaPo5ub6yW0jSzlOpyWzOuWWAW6XNrhgSQWwMezMO5r6z+LX/BSi61L4XeFvhB+zpp+p698TfEOiWdjf39vavnTbqSBBLFbpjdNcBi4yBsQjILkYHUaT/wS6+HtjObjxN8cdf1i3UZMGm+H4rKVgB0Ekk8oH/fNek/Dn4P/AAk+Ac103wp+H66RqU4Iu9Xv3+1atOpwWQzEDy1bjKRBFOBxXRUzXCxT5Zcz8jJYSo91Y5r9jb9mm3/Zq8C3yeJZ7a7+IvinZN4gmjkEosYlO5LNJP4iCS0jKcF8DLBFNe8w3MFwWEMgYrjP49Kx3uNK1Kx+2QkW04XcA3yOD6Ed6h0fUbey3RzRkb+sg5P4ivlcXXniajqy+7senSgqcVFG1Kb37XGIlTyMfOSee/8A9avRfAcmL/TG9yv6EVwEciSoJI3DKehBzXa+CJMXOm98XKr+b/8A169XhuX+0zj3i/zRxZov3cX5o9jooor6Y4gooooAKKKKACiiigAooooAK8q/ax/5NY+Mn/ZP9d/9IJ69Vryr9rH/AJNY+Mn/AGT/AF3/ANIJ6AP5rKKKKACiiigAooooAKKKKACiiigAr6N/Y2/aI+Jvwu8YL8OfD/g+7+IPhfxfMINS8GpE0zXTFSDNbAA+XMFzlsbWUYfgBl+cq6/4SfFPxb8FfiNoXxO8EXpttX0K6FxHknZMnSSJwOqOhZGHox6HmgD9Z9O/4Jkfs4eDvFmrfHLxdpusXfh6wsm1mPwXME8m2dIzLIsuxz5mMYESv5e4HlkIUeqD44eONXsdP8FeCvhoNEiksLbUYDo8cWqKmkSIyRGGAGFQRKFiZc4QbsdMj1n4H/GPwb+0L8K9G+JvhCRZNO1m3xcWkhDvaTgYlt5R03I2QexGCOCK0/Dvws8DeDLXULfwXoUGhPqQ2zT2XEoGScKzZ2qCzEKMAZJGDzWNWFSTXK7Lqerl2MwWHpzliKXPU05b7L11/NST8jyzwd8DvDPhK506TxF599dQrC+iaIs0nnW6LI1ykN0yuUnWKdiEkKhVC8lgcD0zxDLpuk+G77xZ8UdSgttM06N7meHzCLeGPggMB/rH/h/2i2ADkV0VhpmnaDavKZWOxWea7updzkfeYs56DOTgYUc8Cvzh/bA/aal+MGvnwZ4QvGXwdo8xKuhI/tKccea3+wOQg9yx5IC3TpxgrRRy43MMTjantMRK7/Bf1179TyD9qD4/+Iv2gPiLNqU5nt9C05jbaJphORDGcZdgODK5ALHnsoJCivOpvD0djoB1a9e6sb1JfKjilU4mcMM7CBkYU854BTqS21dDTNJis5pvEOqPE0KLK/kbd7KNzR/OhwCCVYYDDnAJQshOFq2o3Ov6mWijmYSSbbeHhnOTgZ2gBnPGSAMnoAMAWch5v8aPCgvkHxK0uH/j7lEOtxoBiO6bJWfA6LNgk+kgfoHQV5LX7YfAH9gbwqPgrrmnfGTSvO8Q+OtN+zTxMo3aPCSrxqnpMHVHZuxVVH3SW/I747/BjxX8APihrXww8YQkXemS5guApCXdu2THMnsw/IgjtQJHAUUUUDP1n/4Ic6u03hn4u6DvYizvtKuwu0YHmpdLnPUn9x07YHqa/UGvyH/4IgassPxG+KGhFk3XmiWV2AW+Y+VM68DuP3/J9x61+vFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFYXiKAF45iMh1KN/n8a3ao6zD5tg5HWMhxWlJ2mjOqrwZ5HdxTQTtDHt+STa2709veop7eC6QRzxh1B3YPrWtr8PlX5cdJVDf0/pWY5UIxdtq45OcY/GvzLG0HQxM6PZv/gH02Hqe0pRn3Q0QRLM04QCRhtLdyKjiu0muJbZUcNFjJI4p1rDHbwLFE7Og5DE5znmpa5jYK4/xJYnUb+QNbNC6KNkp6P149+1dLqFib5EQTmPac8DOf8D71V1xPlhk9Mqa0pO0roiaujzWRTFNJbvw8RwwrV0jULFITb3SIOSQ5XIP1rXurC2u0KSRgE87gMEGucvNJl07gIDFnhlHH4+ldqkpKzOdqxLqEtohzpV4NkuRIsbZAIrDn/ay+GPwn8d6Z4E+Jtxd6Ibm3i1G21V4mks2UyMhV2XLIwMec4K4IJYVagl8xBII3Tno64Ncb8Y/jT8GfDDWfhP4xWymz1e2aUJc6a13ayhW2kEKrENz/d4z1r28h0xiXdM8zNZNYZySvZo+7vD3iTw94u0e28Q+Fdd0/WdLvUElte2FylxBMp7rIhKsPoa0q/nc8L/Fr4i/Df4963B+wv4k8bQaPf6i0ukaVawyXTXcQUEiSzYOsqrhgDIpYIASQc196+Av+CqnxW+GRs9H/bJ/Zx8RaCkxESa/pemy2qyn1NtcYV+OWMcvY4TtX0xxqSdj9LqK8k+Dn7WP7O/x7ihX4X/FbRNUvplyNLlm+y6guBk/6LKFlOOfmCleOCRXrdBQUUUUAFFFFABRRRQAV5V+1j/yax8ZP+yf67/6QT16rXlX7WP/ACax8ZP+yf67/wCkE9AH81lFFFABRRRQAUUUUAFFFFABRRRQAUUUUAfYX/BNz9rY/s8/FQeCvGOpeV4D8azpb3rSviPTrv7sV16KvRJDx8pVif3YFfuACGAZSCDyCO9fy+1+j3wD/wCCjHixv2bk+DlzJcP4+0hP7NsdbkcEnTQuFkznJnjH7sH02uSWByCaPoD9t79qAytefBT4f6iQikw+Ib+B/vEdbVWHof8AWY/3P7wPxnpOlS3xkunika1tdrzlRgsu5QVU9N2CTgkcAntRpel3Wt3UgWZVCbZJ5pW+4rOqFz3IBcE+2TV7xPqFpY2f9gaS2YraHbcyA7lkkBy204BIJA6j+FR0XJBnDeI9bk1W/uDBPILWWXzjErERGTGC4TtnJ568npnA+5P2CP2Ttgsvjx8R9MyzATeGtPnXp6Xjqf8AyGD/AL/9w15Z+xL+yjL8Y9fT4heObB18FaPP8kUi8atcKc+UM9YlON56H7g/iK/qRHHHDGsUSKiIAqqowFA6ACgTY6vjr/gpF+yWv7QHwubxt4R04P438HwvcWqxr89/bdZLc+p43L7jHAJr6o1Dxv4a05FkfU0mBkaMi3/elNuN5O3oFzkms7wNq2u3sF6dctJltAxmgubggfK3JTJxvUdnHykfSgR/NY6PG7RyKVZThlIwQfSkr6e/4KIeGPgt4e/aK1a6+DHinT9St9ULXWsWFiC0WnXpb94qyAbG3HLFQSVbdnGRXzDQUfff/BFzV2sv2pPEWmMz+XqPgm6UKoGN6Xdm4J74Chxx3I/D9ra/Bv8A4JNaqunftr+FrNigOqaZqdou7OSRayTce+IT+Ga/eSgAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACmyIJEaNujAg06igDzbxPbFYkkI+aFyjfj/wDXFcvdNFsWKZHZZmCYUHv9Oleh+KrPcbmMD/WJ5i/X/wDWK4KvkOJqPLi1WX20vvWn+R6mUzvRdN/ZZC0TRWphtAEZUIjz0B7VHp4vRAft3395KjIJC9skVMJSZzB5T4C7t+Pl+mfWkk8ufzLV1fBXDHBAIPoa+cPUJazPEUdy+mObQqJEYEZ/L+tWBEk9tJZBJYljxGGYdQO49RRJbCLTWtlZm2RkAnqcU4uzTE9Uc6pZYwZSMhfmPb3pEeK4j3oVdG/EGnkAggjINMzBaxD7kUa/gBXSYGNf6Dbwh7mzIi3NuZT0JPHHpXxL/wAFC9Nmgl8D3rxOpxfQPk8D/UFeP++ufavvasvxL8LvAPxZ0ifw18QPDNpq9mULRGVcS27HjfFIMNG3upHvxXsZHVax1NPz/JnDmMb4aX9dT8Qfgz8ZPF3wA+J1n8R/BYtZL/T2kheC7jLw3ELgq8bgEEAjuCCCAa+0fGP/AAVP8FeP/hpq/hDxX+z+15datZSWktrPqiTWLMy4DkmIOMN8wAGQQMMDyNb9oD/gjp47sYJPGX7OvimLxLaXEf2ptA1eVLe/jLDdtin4il5P8flEerGvz28b+AfG/wANdfn8K/EHwlq/hzV7fmSy1O0e3lA7MFcDKnHDDIPUE19i1qeNKhTqWlJancfs1fADxR+0h8SP+EC8J6/pujXVrYyarLd3ruAkcbxqdgQEs+6RcDjgE54r72l8F/8ABST9lHwpc+KfAnx7tviHoWj25ub3RtUMl7IkMYLHykuVLbFUElYpUbsFNfmFoWv674W1a21/wzrV9pOp2b+ZbXljcPBPC3qroQyn6GvVPEX7Yf7TXivwxc+Dtf8AjHrt1pN5Aba5hBjjeeJgVZHlRQ7KwJBBY7gSDmkE41udODVj7v8Ahf8A8Fu3AhtPjR8FATx52oeF73H122twf5z19YfDj/gpp+xr8RxHEnxWi8NXknJtfElrJYbPrMQYPykNfjj+yB4v/Z38IfEua6/aQ8IjWdCubMw2k8sD3MFjPuB8yS3XmRSoI4DEH+E5yv1B+09Z/wDBNvxH8KNZ1j4fa1oGleLbeyaTRF8NwzRyT3G0mOKWAKE2MQAzMoKgk5B4IE63LNQ5X6n6+eGPGXhDxtYf2r4M8VaPr1kcf6Tpl9FdRc9PnjYjse9bFfzffstfAH4p/H7xxeaT8LPElp4fvNGtPttzqU97LbeShYKApiBcsT6DHHJHGfrHx34Z/wCCnX7LHhK68b6R+0NqPijQNIi867aPU31V7KFcZdoL+Jv3a99m7CgkgAHAW61NS5G9T9kKK/EHwR/wV8/bHsbm3066sfB/i+eZwix3ehSLPKSeii1kjG7sPlP0Ne1f8PjvjH4KWAfFX9lD7KZsAN/aF1pYc4H3RNbyZ7nGenfvQXdbH6qV5V+1j/yax8ZP+yf67/6QT18VaR/wW9+Fk2P7f+B/iqz65+x6jbXOPT7wj/z61T+Mv/BXX9nP4nfBLx58OtM8EfEay1fxV4V1LRraS5sLH7NFPcWssSF3W7LbNzjJCE47UDPyNooooAKKKKACiiigAooooAKKKKACiiigAq5o+rXuhanbatp8uy4tnDoex9QfYjIPsap0UAfYXgn4i3Gr+E/M0i9mS2vQGMQlObaUFd+B2b5QCeMgKfSvWP2df2ftZ+P3jA6MpltPD9iFk1nUFH+qjOcRoTx5j4IHpgsc4wfi34FeJItP8b6d4Z1fWLfTNH168hs7i8umIgsS7hRO57Kufm/2ee1fudfaJH+zD8CVtfhfoSahDpm2bUb94/NkVGx5188KlTOVX5vLDr8qgbgFqZSUYuT6G2GoTxFaNCnvJpK+m56roGg+GvAPhe08P6HZ2uk6JotqIoYwdkcESDklj+JLE5JySckmuK0b4zeE/iFqN3oPhJo9Wt9v2eUyLLESz/cfbsyYHU5WZQVODg157ceEvi58bLJbjWfH0ek6GEt9Q07UtKAjtp4pbcb/AJYbksJ4ZcuHMjxHjAIyK8M/aE/b2+Bn7K1rqXw//Z+0bT/EnjF5JBczo5ksrF2dnKvKD8213YiFCFXJHy5rKE5zldK0fM9LFYLB4Wg4VKnPWdrKPwx/xNrV26LZ3ufTPxD8afC39nrwi/iv4zeNLa30yxkZtPtJXMkshGQioh+aV9ny4UBcDpxmvyz/AGtP+ClfxN+OrXfg74cNdeDPBTExlIZdt9fp/wBNZF+4p/uIfqTnFfMfxW+MPxH+Nviqfxl8TPFN5rWoykhDM2IrdCfuRRj5UUegHbnJ5rja3PGsFFFFAz6P/wCCc+rnRP21vhXeB3XzNUmtMqRn99azw457fvOfav6Ga/mv/ZK1ddD/AGpPhHqbsixxeNtJWVnBIVGu4kc8c5CsSPf1r+lCgAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigDH8Qw5jinA+6Sp/H/8AVXmV5D9nupYeyOQPp2r1vUofPsZUxyF3D8Oa8z8RQ7LxZgOJU/Ucf4V4vEVH2mCjVW8H+D/4Njpy6fJiXH+ZfijKoqK4uYrVBJMSAWC8DPJp06yPE6RPscjhvSvhz3yO7+17V+xhN24bt/pUiMZUbdGyclcN396IFkSFElk3uBhmxjJp9AHGxm6iluBemNY0fCHPbJ/+tUrxxTptkVXQ84PINTataxy3E8EgO1m3cHHvVQzRWzw2ixvhhtXAyAAO9dSd9Uc70JlZWGVII9q1NAOL1h6xkfqKyILeG2j8uFdq5zjOa09EbGoIPUEfpXflcuXG0n/eRzYxXw8/Q918ONv0OyP/AEyA/Lisb4j/AAo+Gvxf0FvDPxP8D6N4m00klIdRtVl8pj1aNiN0bf7SkH3rT8Ivv8PWh44DD/x9q2K+7n8TPFh8KPza+OX/AARd+GviI3Gr/ATx3e+EbtsumkaxuvrAnHCJNnz4h05bzj7V8CfGb/gn/wDtWfA4XF74n+F17qukW5OdX0A/2jalQcb28seZGvvIiV/RBRUln8q5BUlWBBBwQe1Ff0ofFv8AZS/Z1+OYll+J/wAI/D+r3sww2orb/Zr/AP8AAqErL+G7FfHHxR/4Ip/CLXDLefCX4oeIfCs7ZZbTVII9TtQf7qkGKRR05ZnP1oA/JPwV478ZfDjxBB4q8B+JtR0HVrcFUu7Gdon2nGVOPvKcDKnIPcGvU/G37bP7UPxE8J3vgfxd8V7u70XUYvs93bxadZ2zTx4wUaSGFXKkcEbsMODmvZ/iN/wSJ/a58Fmefwxpvh3xtaRksjaRqiwzFfUxXIj+b/ZUt7Zr5i8ffAj41fC15B8RfhR4s8OpH1n1DSZ4YCPVZSuxh7gkUEuEW7tHQ/sv/tC6j+zP8UE+I1h4YstfSSxk025s7iQxMYpGRiYpQG8uTMajdtbgsMc19Z/F3/gqN4J+Ivwy1zwRB8DJrm41ywks2XVr6KW1gZ1I8zaqbnKE7l+4cgHKmvzxoosZzw9OcueS1PUv2arP4FX/AMWdNt/2ir+9tfBxjkMjW/mBHn48tZWiHmLEecsnIO3kDJH3N8X/AA//AMEsrb4d372d94djvVs5P7Pfw1f3E+oCXb8mFVmBbJA/fDb1zjBI/MeigVSjzyUuZoKKKKDc/VD/AIIk22m+INF+NHhvXdKsNQsUm0S48i6gWVXZhfqcq2QQNi447mvunxh+xH+yT46D/wDCQfs+eCw8gw8un6cunStznJe28ts++c1+ff8AwQ+vkj8efFTTShLT6Rp84bPACTTKR+PmD8q/XOgD4d8Zf8Eef2RfEjSSeH18Y+E2YlkTTdZE8aegxdJKxA/3s+9eF+M/+CHrhZJ/h7+0AC3/ACztdZ0PA/GeKX/2nX6rUUAfz1/tT/sCfGn9knw9p/i/x/qfhrVdD1TUv7Lt7vR7qaQiYxvIqyLJEhUlInPGfunnvXzXX7l/8Fg9B/tf9j2bUPLDf2H4m0+/zgHbuEtvn2/4+McetfhpQAVPc2F9Zxwy3dlPAlwu+FpI2USLxypI5HI5HrUFf0f/ALGd8mufsjfCKWZop0/4QvTrRgACpEdukW0jvjZg+4NAH84FFf1Aap8Lfhlrm4a38OvDGob12N9q0i3l3L6HchyPauR1X9k/9l7WzI2p/s6/DWaSUgvL/wAItZLKcYx86xhugA69OKAP5raK/oj1T/gn3+xlq5Y3f7PvhmPc/mH7KJrbnnp5Trgc9OntxX5nf8FV/wBlj4Qfs2698O7v4N+DR4e03xNbagt5Ct/dXQkngeA7szyuV+W4AwuBx+QB8G1+qH7Df/BQ/wCG2lfAa/8Ahv8AtIeI4oLrwbaC306W6ha4bWNPKlVg24O+SMDZtP3kKdSGNflfXX/B1NMk+LngiPW9Oh1DTm8R6eLu0miWSO4h+0R742RvlYMuQQeCDg0Bs7o+n/2sv+ClXxL+Of2rwZ8NftPgvwQcxeXBJtvr+Pp+9kX7ikfwIfqSDivmzwt8BPjl44ZB4O+DfjbWw5wHsNAup0HOMllQgDPcnAr+kbwt8L/hn4GCL4K+Hfhjw+IxhBpekW9ptGMYHloMccV09AH8+fhX/gmp+2v4t2SWvwQvtPhfky6rqFnZbOM8pJKJPbhT717H4U/4IvftMavtl8UeM/AegRHqn2y4upx/wFIQn/j9ftTRQB/N5+1l+zZq/wCyn8W5fhTrHiSDXpF0631GO/htWt0kSUNwEZm6MjDOecduleNV+kX/AAW58K/Y/jD8OvGoiIGreG5tM34OGNrctJj0yPtg/Me1fm7QB0fw01ZtB+I3hXXUZ1bTtbs7sFG2sCkyNwex461/ULX8q9f1KeF9XXxB4Z0jXldXGpWMF2GVSoPmIrZAPIHPQ0AalFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAJ14Nee+K7Py42wOYJcZ/2Tx/hXodcx4qs/NMgA/18XH+8P8AIrPEUfb4epQ/mTt6rYSn7OpCp2f4HndRzzxW0fmzMQoIGQCf5VQutZEbGO3j3EcFm6flWdPfXVyNsszFT/COB+VfmqpvqfTOa6HQTXFvbKz3FxHGFGTlucfSs19Q0m6lgv4rqaQJkqEXAP51lVPBp97OALaymkHbZGSP0rWFG7tFXZnKpbVuxBrWsQLdBktXbcoOWfH9KyptcuBGxt7OEvjgMSc/rW1qfhLXZhDL9hMa5IzIwX9OtNtvA1w2Dd3saD0jUt+pxXr4fKsVUS5ab+en5nDVxtGL1n/XyMSHW75olM1vbq5HzAA4B/Otnw1dXt5qMbC0HlpnfIuQF4PrW3YeD9KhdQLeS6kPTf8ANn/gI4rsNM8G6tcqqraraQjoZBtwPZete3g8hlTqRrV5JWadl5f15nn1sx54unSTd9DrPBL7tBjGR8sjj9c/1reqjo2lRaNYrZxSNJyWZj3J9u1Xq9mbTk2jGmmopMKKKKksKKKKACkIDAqwBB4IPelooA8u8cfst/s4fEjzX8a/A7wVqc8xy92+jQR3RP8A13RRIOv96vBvGP8AwSU/Yy8U+Y2l+E/EHhaSTcS+j65McE9wtz5yjHYAY9sV9lUUAfmX4q/4If8AgG63nwR8efEGmdSi6ro8N96YBMbw+/OO44458i8Sf8ES/jraFj4T+LXgXVFB4+3rd2TMOeyRSjPTv361+yNFAH8wnxa+GHib4L/EfXvhd4y+y/214dufst2bWRnhLbQwKMyqSCGBBwK5KvrD/gqXoY0X9t7x9JHGUi1OPTr5Bz/FY26scnrl0c8cc47V8n0Afon/AMETNQEf7QHjnS/tBU3Hg5rjyucP5d5bLu9OPNx/wKv2Vr8SP+CMt/JZ/tZatboExfeCr6B93XAubOTj3zGPwzX7b0AFFFFAHzL/AMFKNCHiD9iT4n2mwFreztL5TxlfIvbeUkE+yEeuCR3r+fGv6U/2rtD/AOEk/Zi+LGigZkuPBeq+UM4HmLayMmeDxuVa/msoAK/oY/4Jzal/av7FHwrujJE+zTJ7bMZyP3V1PFj6jZg+4Nfzz1+93/BKbUjffsQeCLYtEf7PvNUthsPIzf3Evzc9f3vtxj60AfXVFFFABX5w/wDBbjw8Ln4KfD7xZsydN8Uyaduz0+0WskmOv/Tr6duo7/o9Xxr/AMFbPDn9t/sW+INT25/4R/WNN1HOem64W29R/wA/Pv8ATuAD8Ias6ZfPpmpWmpRoHa0nSdVJ4YqwbH6VWooA/qkgnhuYY7m3kWSKVQ6OpyGUjIIPpipK5P4Sasde+FPgvXWlMh1Hw9YXZcqFLb7eNs4HA69K6ygAooooA/Nn/gt34WF38JPhv428oE6V4juNLD4GR9qtzJj15+xfp9K/H2v3k/4Kx+Fm8RfsWeJ9QSMyP4d1PTtVUKCT/wAfCW7HA9FuGJ9ACe1fg3QAV/TB+zVrDeIP2dPhbrrsWfUPBekXLln3Hc9nCxBPc5Jz71/M/X9FX7A+rjW/2OPhNeK6sI/D0VplVKj9yzw457jy8Z79RQB79RRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABWfrFnJdwKYV3OjZxnt3/pWhRVRk4u6JlFSVmcVP4dimctcaIkjHqzW4Y/nimJ4WtRyugQj62y/4V3FFU5wbu4L7jNUmtOZnJweHpk/1OnRx/RVWraaDet95o0+pzXQ0VSryStFJC+rx3Zz9z4SivYhHcXjABt3yL/jT7bwZoVuQWgknI7yuf5DArdoqXVm+paowXQht7O0tF22ttFCP9hAv8qmoorM0sFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAfiR/wAFmNCOl/tX6XqixEJrHg6zuS4U4Z0uLqIjJ4yBGvTsR618HV+mX/BcDQfs/wAQ/hb4n8sj+0NGvrDfg8+RNG+M9OPtP6+4r8zaAPsv/gkhex2n7aGgwOjE3mjalAhHQEQGTJ9sRkfiK/d2vwB/4Jd6h9g/bl+G+648qK5/tO3k9H3abd7V/Fwv44r9/qACiiigDM8T6MniLw1q3h+QjZqdjNZtkkDEiMnUc/xdq/lskjkhkaKVGR0YqysMFSOoI7Gv6pq/mK+NWg/8It8ZPHnhgxmP+yPE2o2Gwgjb5VzImMHkfdoA4yv3F/4I7agb39kFrY7P9A8VX9uNp55SCTn3/eflivw6r9nP+CJ9+ZP2cfGel7Vxb+Npbjdnk+ZZWa4x6fuv1NAH6GUUUUAFeEft2+HT4o/Y9+LemqpJh8M3Go8f9OwFz6j/AJ4//r6V7vXNfEzw4fGHw38V+ElQsdb0S804KO/mwPHj/wAeoA/l8ooooA/pJ/Y+1X+2v2U/hDflizf8ITpULsz7yzR2scbEn1JQmvX6+bf+CcmrLrX7E/wsvFZGEemz2mUBA/c3c8Pfv+75984r6SoAKKKKAPG/2yfC48Zfsp/FjQfLDu/hK/uoVOPmlghaeMc8D54l57da/m7r+prXdItvEGiahoN7n7PqVrLaTY/uSIVP6E1/Lfqum3ejapeaPfxmO6sZ3tpkII2ujFWHPPUGgCrX76/8EsNXOqfsP+AYXdnfTp9TtGZn3E41C4dR7AK6gD0Ar8Cq/b//AII3auNS/ZHu7PeG/srxffWmNuNuYbabHv8A67OffHagD7pooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKAPzK/wCC4OhC48BfCvxN5ZJsNXv7Hdszjz4Ynxuzxn7N0749ufyMr9tP+Czehf2n+yno+qog36P4ys7hn2ZIR7e6iIzngFpEPfoPqPxLoA+iP+Cel9Jp/wC2h8KZ4kVi+tNAQ3TEkEsZP1AYmv6IK/m6/Y2v49O/ay+D9xKHIfxrpcA2dcyXMcY/DLjPtmv6RaACiiigAr+c39u3Qf8AhHP2wvi3p/lhPO8TXN/gAD/j4IuM8evm5/Gv6Mq/BH/gqzoQ0f8Abb8Z3aoFXV7PTL4AYx/x5QxE4HqYSTnnJJ70AfItfrn/AMEPr4yeAvippvlgCDV7CffnrvhlXGPby/1r8jK/Ur/ghrfCPVPjJpnlkmeDRZ9+emxr5cY9/M/SgD9XqKKKACiiigD+YT4v+Gz4O+LXjbwiyFDofiK/00qRjb5NxJHjoP7vpXJV9Aft++Gz4U/bK+LGltGUM+vvqWCMf8fSJc56Dr52fx6nqfn+gD95f+CTertqX7E/hWzLORpWp6naAMAAM3ck3GOo/fd++favsOvgn/gjDqv2/wDZT1uwZhu03xteQhd+Tta1s5Acdhl2H4GvvagAooooAK/mw/a18LL4L/ae+KnhuOMRw2vi7UWt0AACwyTvJGOOPuOtf0n1+B3/AAVT8Ljw3+2v4zuY49kOu2un6pGvP8VpFG559ZInP40AfJFfsV/wRH1jz/gp8QdA8wn7F4pjvNnmZA861jTO3tnyOvfHtX461+qn/BDbVxj4xaDI4znRruJdv/X6rnP/AH74+tAH6qUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFAHyX/AMFUNCOtfsReOp0QPJpVxpt+g2bjxfQIxBzxhJGOfQEd6/Auv6NP259CHiL9kD4uWBj3+V4Wur7G3d/x7r9ozj28rOe2M1/OXQB6F+zrqDaT+0F8MdVS4S3az8ZaTcCV8bYyt5E2454wMZ54r+mav5dvh7ff2X4+8Nan5Xm/ZNYtJ9m7bu2zI2M84ziv6iaACiiigAr8Vv8AgtLof2D9p3w1rUY/d6p4LttxJ5Mkd1dqe3TaY/1r9qa/Jr/guNoRh8Q/CPxMqEi7s9VsXYZ48p7V1B7DPnNjucH0oA/Lyv0g/wCCI1+I/jJ8RNL2tm48MxXGc8Dy7lFxj1/e/oa/N+vvn/gi7qQs/wBqnxBZO0u2/wDA95Eqqfl3rd2TgsM+isM89fc0AftfRRRQAUUUUAfhR/wV18NjQ/2zNW1MIF/4SHQtO1InH3tsRts9P+nbHfp+A+Lq/SL/AILdeHRbfGD4c+LfLwdS8NTadv8AX7PctJj8PtX61+btAH7Af8EQ9XE3wo+JOg71Js/ENtdldpyPNt9uc9CP3HTtg+or9KK/Kb/ghtqzLc/GLQ3LlXTRruMcbVIN6rfidyf981+rNABRRRQAV+Nf/BbHwv8AYfjz4H8XpFtTWPCn2JmHR3t7mZienXbcoOp4A4Hf9lK/Mb/guD4YW58FfCvxmsYDafql/pbOAORPFFIAT/27Nj6n15APyQr9IP8AgiNrHkfGb4h6B5hH2zwxFebPMwD5NyiZ298ef17Z96/N+vur/gjZq4039re9s94X+1fB99aY253YmtZse3+pzn2x3oA/b6iiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooA4z41aJ/wk3wb8eeG/KEn9reGdRsdhQPu8y2kTG08H73TvX8xVf1SzQxXETwTxJJFIpR0dQVYHggg9RX8tXiHSJvD+v6noNwT5um3ktm+QAd0blD0+lAFGGaW3lSeCV45Y2Do6MQysOQQR0Nf1QWl1BfWsN7ayb4biNZY2wRuVhkHB5HBr+V2v6hfhvey6n8O/C2pTqiy3ei2c7hAQoZoUJxntzQB0dFFFABX5vf8FudB+0fBv4d+J/LJ/s/xNLYb8Hjz7Z3xnpz9m/T2NfpDXxF/wWD0H+1/2PZtQ8sN/YfibT7/ADgHbuEtvn2/4+McetAH4aV9of8ABIvU/sH7ZmkWvmSL/aWhaja4Xo2IhLhvb91n6gV8X19Uf8EvtTGmftyfDZnmaOK5bUrZwBnfv066Cg+2/Z+VAH9AFFFFABRRRQB+Zf8AwXB8Om5+H/ws8WbR/wAS3WL/AE7d6faIYpMdf+nX0PTqO/5F1+43/BYnw8da/ZBGpbSf7B8VWGoZHbck9tz/AOBNfhzQB+jf/BEjVRD8b/iBom5M3fhVLoKX+Y+VdRLkDuP33J7ZHrX7G1+IP/BGzVxpv7W97Z7wv9q+D760xtzuxNazY9v9TnPtjvX7fUAFFFFABXxH/wAFgvDH9v8A7Hk+q7Sf+Eb8S6fqeeeN3m2uf/Jrv619uV8/ft/eGD4u/Y1+LGlCPzPI0B9Uxtzj7JIl1noenkZ9sdR1oA/nYr62/wCCVesf2X+2/wCBbdpCianbanZsfM2g/wCgXEgB9ctGoA9celfJNfQH7AWr/wBiftlfCe8LhfM19LTJXd/rkeHH4+ZjPbrQB/RRRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFfzSftP6F/wjP7SXxU0ARhEsfGerRRAKFHli7l2EAcAFcHHbNf0t1/PH/wUT0L/hHf21Pirp5jCebq0V9gKF/4+LaG4zgevm5z3zk80AfOdf0wfs1XsWp/s5/CvUoFdYrvwTo86BwAwDWUJGcd+a/mfr+kH9i6+k1H9kr4QXEk6zFPBumwblxgCOBIwvHcBAPXjnmgD2eiiigAr5l/4KUaEPEH7EnxPtNgLW9naXynjK+Re28pIJ9kI9cEjvX01Xk/7Wehr4j/AGXvi1o5GXm8F6q0QyAPMS1kdMkg8blWgD+a+voD9gLUzpH7ZXwnuhOIfM19LXcVznzUeLb0778e2e1fP9es/skaidJ/an+EF950cSr450dJHkICqjXkSuST0+Vjz2oA/pPooooAKKKKAPm3/go54dPif9in4paesZZrfTYNRG1ckfZrqC4J6HtEc+2elfz01/TL+0T4dHi/4AfErwt5YdtW8I6pZoCucO9rKqkcHkMQRx1Ar+ZqgD66/wCCUuqNp/7b/gm0DSAanZ6panaeCBYTy/N7fuvzxX73V/O7/wAE+NTXSP2zvhTduYwJNbNr+8baMywyxDn1y/A7nAr+iKgAooooAK5f4peGT41+GXi7waIxIde0K90zYRnd50Dx46H+9XUUUAfyr16Z+zDq50H9pP4U6yHZVs/GukTPtfZlBeRbhnsCMg+xNYnxo8NDwZ8YvHfg9YxGNC8S6jpoQfw+TcyR46n+761i+D9XXQPF2ia8zqi6bqNvdlmUsAI5FbJA5I46CgD+pGiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAK/CP/grfoX9k/toa7qHlhf7a0XTb7IUDdtgEGSR1/wBRjJ54x0Ar93K/GL/gtdof2X9ovwb4gSMKuo+DY7diEADvFd3RJJ7nbKg57AUAfnpX9D3/AATyvhqP7F3wquFjKbNHeDBOf9XcSx5/Hbn8a/nhr9/v+CXeof2h+w18N91x5stt/advJ6pt1K72r+CFfwxQB9VUUUUAFZfirRh4j8Max4eYgDVLCeyJLEAeZGydQCR96tSigD+Vh0eN2jkUqynDKRgg+ldh8GNQOkfGHwLqq7M2XiXTrgbzhfkuY259uKb8YtCPhf4ueN/DTIUOk+I9QsSpzx5VzImOef4e/Nc1pV8dL1Sz1NYxIbS4ScITjdtYNjP4UAf1PUUUUAFFFFAEdxbw3dvLa3MYkhmQxyIejKRgj8q/ls8SaNL4c8Rap4euG3S6XezWTnjlo3KHp/u1/UvX82n7X3h//hF/2p/i1owQpHH4x1OaJTn5Y5bh5UHJJPyuvPegCv8Asoat/Yf7UHwk1QyeXHB430jzW27sRm8iV+P90mv6U6/l8+GWqNofxJ8J60jSK2n65ZXQMZw4KTo3HvxX9QdABRRRQAUUUUAfzxf8FEPDR8KftpfFTTSuBcaumpj3+1W8Vz6n/nt/9YdB86V9x/8ABYrw1/Yf7XaasqEDxF4WsNQLdiyPNbfni3H6etfDlAH9RngXVm17wR4e112dm1HSra7JdtzEvErcnueetbleW/sr6uuu/syfCbVgyFrnwTpDSbAQof7HEHAB5wGBH4V6lQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFflF/wAFx9D2aj8IPEkcQPnQ6vYyuEHGxrR0Bbqc+ZJgdsH1r9Xa/OD/AILcaEbj4LfD3xNsBFh4oksd2zOPPtZHxuzxn7N0749uQD8dq/d3/gklfG7/AGLtAtzGF+xazqUAOc7szmTPt/rMfhX4RV+3n/BGu/W7/ZJv7cXLSGy8Y30BUkkRZgtZNoz0/wBZu4/vH3oA+7KKKKACiiigD+cr9ubQf+Ec/a/+Lmn+WU87xTd3+CCP+PhvtGefXzc/jXhlfWn/AAVR0H+xP23fHNwqbItWt9Nv0G3A5sYI2PXnLxsc8ck+lfJdAH9RXgG/Gq+BPDmqKrKLzSbW4AY5I3RK2CfXmt6vOv2cdSGs/s8/C/WBJLIL7wZpFzvlOXbfZwtlvfnmvRaACiiigAr8A/8AgqJ4dbw9+258QGWMpBqg0/UYcg/MHsoA56Af6xZOmfrnNfv5X4of8Fn/AA9/Zf7Umha3HGBHrXg61kdsAbpY7m6jI98IsfPv7UAfA1f1M6Dqia3oWnazGYyt/aRXSmNtyEOgbg9xz1r+Wav6af2fdWGvfAX4ba6sgkGo+ENKuw4UqG32kTZweR16UAd/RRRQAUUUUAfkj/wXC8Mm38Z/CrxisRxf6ZqGmO4HTyJYZFBP/by2Poa/MWv2R/4LZ+GvtnwK8CeLVi3NpXis2JYDJVbi1mYnpwM2yjr1x7V+N1AH9En/AAT71VtZ/Yy+FF47OTHoYtPnxnEMskI/DEfHtivoSvkT/glLqy6j+xD4Ks1KE6Xe6paNtzkE3083PviYfhivrugAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAK+IP+Cwuh/2t+x+9/5e7+xfFGn32dm7buE0Gc/w/wCvxn3x3r7fr5d/4Ka6H/bv7EPxKhSMtLaQ2N8hCFivlX1s7HA/2A4z2BJoA/n5r9nf+CJ988n7OHjPTSgCweN5pw2eSXsrNSPw8sfnX4xV+vv/AARCvRJ8L/iZp32ksYNftZ/J3HCb4CN2Ogz5eM/7PtQB+ldFFFABRRRQB+J3/BZ7Qv7N/ao0PV0jAj1fwZaSs4UDMiXN3GQe5IVU5PqB2r4Ir9OP+C4ehm38bfCjxL5YAv8AS9Rsd3HPkSwPj14+09+OeO9fmPQB/R7+xTqX9q/sj/CG6+0eds8H6fbbsYx5UKxbfw2Y/Cva6+b/APgnPqLap+xT8K7lnjYppc1vmPp+6up48fUbMH3zX0hQAUUUUAFfk9/wXG8P+VrPwj8VIhP2m21XT5W5+Xy2tZEHXHPmv27H2x+sNfnX/wAFsvDv2z4B+B/FSxhm0vxaLIsBkqs9rOxPTgZt1HUckdewB+Ndf0c/sRaqdZ/ZE+EV4Wkby/CVjaZkxn9zGIe3b93x7Yr+cav6Bv8AgmTqg1b9h74ZTfKGghv7VlD7ivl6hdIM+hIUHHuKAPqKiiigAooooA+PP+Csfhsa7+xV4p1DYGbQNS03Ul4yRm6jtyRx6XB9OM1+DVf0cftv+G/+Er/ZE+LekiPeY/Ct5fquMkm2T7QMe+YRX849AH7d/wDBGnVm1L9krUbMs5GleMr60G4AAZt7SbjHUfvu/fPtX3bX5vf8ERtWE3wb+Imhb1Js/E0V2V35I822Rc7e3+o698H0r9IaACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigArxv8AbK0T/hIf2T/i7pvl+Yw8G6lcouCSzQ27zKAByTmMY969krkfixaWOs/DPxd4durm2T+1NCvbLbLLsB8yB0wSOR97tzTSb2E2lqz+YWv1e/4Ia3/maV8ZNM8rH2e40Wffu+9vW+XGPby/19q/KGv0u/4Iq+L7Dwzr/wAWbTUbp0ivLPSpBGibi7RvdDP4CQ/99U4QlN8sVdk1KkKcXOo7JH69UVxUvxY8PJxFZ38h/wBxAP8A0KqUvxetR/qdDlb/AHpwv9DXUsBiXtA4JZxgY71F+L/I9CorzOX4vXR/1OhxLz/FOW/oKqyfFrXT/qtPsF5/iVz/AOzCtFleJfT8TCWf4FbSb+TPif8A4LfaJ9o+Gnwx8Sbf+PHXbyxzxx50Cv8AX/l3/wAe1fkLX6//APBWDxBq/jT9mWx+3x2yR6T4rs78eUpU5MNzDjknI/fdPYelfkBXLXoToy5J7nfhMXTxVP2tLbY/fL/glXfG7/Yd8BW5jC/YrnVYAc53Z1G5kz7f6zH4V9bV+bn/AAS68d+ILD9lqDS9Lv2ghstfvoypiRsk+W+QSCcfOP1r63b4g+MHGDrB/CGMfyWuylldapBTTVn6/wCR5uIz/DUakqUoyutOn+Z7ZRXiH/CeeLv+g1J/3wn+FH/CeeLv+g1J/wB8J/hWn9j1v5l+P+Rj/rLhv5Zfh/me318b/wDBWrw//bP7FniPUdgb+wdW03UM8fLuuVt89f8Ap4xxnr+I9Z/4Tzxd/wBBqT/vhP8ACvFv20tX8S+L/wBlX4l6Feag08B0R7x0aJOfs7pcZyFyMeVn8Oo61M8prRi5XWnr/kXT4iw05qCjLV26f5n4X1+63/BInVhqP7GmlWYdG/svXtRtCFBBXMgmwfU/vs/Qivwpr9a/+CSvxD17Sf2ePEGhWV0nl2ni64kVHhUhVe1tD16n5lbr61w0KEq0+SG562LxcMNS9rUvbyP09oryFPil4nUYYWb+7RH+hqdPi1r4z5lhp7emEcf+zV2PKsR5feeYuIcE9218j1eivMY/i7fD/W6LA3+7KV/oatxfF+A/67QpF/3bgN/7KKzeW4lfZ/FG0c8wD+3+D/yOk+IHh4eLvAfiTwmU3jWtIutO25xnzYXjx1H971H1r+Xav6cYfizoLYE1jfRn2VGH/oVfzd/F7R4/D3xY8a6BCGEem+Ir+zUMMECO4kQZ5PPHqa56uHq0v4kbHbh8ZQxF1RknY/Sf/ghtq4x8YtBkcZzo13Eu3/r9Vzn/AL98fWv1Ur8cf+CKfiWy0j4u/EPSr+9W2ivfDkM4Mku2MtHcqoyDwTiY4PYZ9a/YS31TTLsgWuo2s2enlzK2fyNZ8krXtobe0hzct9S1RRRUlhRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUVQ1DXtF0oH+0NTt4WH8DON//fI5/SqjGUnaKuTOcYLmm7LzL9FcVqHxV0G2ytjb3F4w6HHlofxPP6Vzl98Vtdnytla21qp6EgyMPxPH6V2U8uxE/s29Ty62d4Klpz3flr/wPxPWKq3eqabYAm+1C2t8f89ZVX+ZrxC+8VeI9Rz9r1m6YHqqvsU/guBWWSWJZiSTySa7YZM/ty+48urxPH/l1T+9/wCX+Z7Rd/EXwpaZA1Bp2HaGNj+pAH61iXfxdsVz9h0eeT0MsgT9BmvMaK64ZVh473f9eR5tXiHGT+G0fRf53O2uvivr0uRbWdnAPUqzt+ecfpWTc+PfFl1kNq7oD2iRUx+IGa5+iuqOEoR2gjgqZli6nxVH99vyLlxrGr3f/H1ql3N/vzM38zVOiit1FLRHJKcpO8nc/n28R6Z/YviHVNHxj7BezW2M5xscr179K+1f+CUNzEnxM8b2ZJ8yXQopV44ws6g/+hivlj9oPTDo3x5+I2mFWC2/irUlTdjJT7TIVPHqpB/GvpP/AIJVXZT46eJ7DYCJvCUs27PTZd2oxj38z9K+Wwa5cXFeZ+g5m+fLpy7pP8j9SKKKK+qPzwKKKKAPnj/goDpf9qfsneNSozJZmyuk5wPlvIN3/jpavxsr9xv2q9M/tf8AZt+JVrz+78M3l11x/qoml/8AZK/Dmvnc4j+9i/I+14alfDTj2l+iP1Q/4JX3vn/ADX7Jpiz2vi24wh/gRrW0I/Ng9fZVfDX/AASiu5H+GPjaxKr5cOvRyqe+XgUHPt8g/WvuWvXwDvh4eh85m6tjqnr+gUUUV1nmhXLfFbRv+Ej+F3jHw+FDf2poF9Z7SM58yB0xjI/veorqaR0SVGjkRXRwVZWGQQexFKSumioS5ZKS6H881fpT/wAEnb9pPA/j/SyX22+q2twAW+XMkTqcD1/dDP4elfnN4i0ptC8Qanoj7t2nXk1qdxyco5Xk/hX3n/wSVvlj1X4naYQu64t9MnHzc4RrpeB3H7wc/T1r5fLXbFRXr+R9/ni5sBNry/NH6L0UUV9SfnwUUUUAFfiT+2Noo0H9p/4jWITb5utPe4/67qs+eg/565/x61+21fkH/wAFHtG/sz9qjXL3YV/tfTrC8yQfmxAsOef+uOOPSvKzeN6KfZn0PDU7YqUe8f1Rv/8ABLzUBZftHX9sSgN/4Wu7cbjycT20nHv+7/LNfq7X4+f8E6NR+w/tW+GrbzNv9oWV/bY253YtZJce3+qz+HvX7B1WUu+H+ZHEcbYy/dL9Szb6lqNnj7Jf3MGOnlysv8jWpbeOPFdrjy9anbH/AD1Ak/8AQgawqK75UqcviimePTxFan8E2vRs7O1+KviKHAuILO4HcshVv0OP0rYtfi7AcC+0WRPVopQ36ED+deaUVzzy/DS3id1POcdDapf1sz2W0+JXhS6wJLqa2J7TRH+a5Fbdlrej6jgWOqWs5P8ACkoLfl1r5/orknk9J/BJr8T0KXEuIj/Egn+H+Z9HUV4FZeItd07H2LVrqIDookJX/vk8V0Fj8UvEdtgXaW12vcumxvzXA/SuSplFZfA0/wAD0qPEmGlpUi4/j/X3HrtFcLYfFjSJsLqFhcWzHuhEij+R/Sul07xR4f1XC2WrW7u3RGbY/wD3ycGuGpha1P44s9ahmGFrfw5p/g/uZq0UUVznYFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUVXvb+y06A3N/dRW8Q/ikYKPp7mmk27IUpKKu3oWKK4PWfitp1vui0W1a6ccebJ8kf1A6n9K4jV/GXiLWty3WoOkTf8sYfkTHpxyfxJr0KOWV6msvdXn/keLis+wtHSD5n5bff/AJXPWtV8X+HtG3LealGZF/5ZRHe+fTA6fjiuQ1P4tscpo+lgeklw2f8Ax1f8a85or1aWVUIay95nz+I4gxdTSnaK8t/vZtal4y8SarkXOqzKh/5ZxHy1+mB1/HNYxJJyTkmkorvhTjBWgrHjVK1Sq+apJt+YUUUVZmFFFFABRRRQAUUUUAFFFFAH4oftp6YNJ/al+IlqqgB9UFzwu3/WxRy9P+B9e/WvU/8Agl1cywftG6jFGRtufCt3E+R2E9s381Fcl/wUPtI7f9rLxdMjMWurfT5XyeARZQJx7YQfrW9/wTMupYP2mo4o1Urc6DexSEg8AGN8j8UH518vTXLjrf3v1Pv6z5spv/cX5H610UUV9QfABRRRQBzvxH0w638PPFGjDOb/AEW8teDj78Lr17da/Aev6GiAwIYAg8EGv59PEGmHRNe1LRm3ZsLuW1O4gn5HK8kcHpXh5yvgfr+h9bwxLSrH0/U/Qn/gkrdxvpXxNsQreZDcaZKx7YdboDHv8h/Sv0Cr85P+CS91Imt/EuyEoCS2umylOMsVa5APrxvP5/Sv0bruy13w0fn+bPIzxWx8/l+SCiiiu48kKKKKAPws/aR0g6F+0F8R9M2lVj8U6g8YOeEe4d06/wCywr6X/wCCUmoeX8XPGOlbo/8ASPDguMH7x8u4iXj2/e8/UV43+3bpI0b9q7x9bqhVJ7m2u1POG820gkJGf9piPTINd/8A8EwNR+xftI3Vt5oX+0PDN3b42534lt5Me3+rzn296+Xoe5jUvNn3+LftMqcu8U/yZ+sNFFFfUHwAUUUUAFflr/wVT0oW3x08N6si7VvfC0UbfKfmeO5uMnPc4dR+A9a/Uqvzb/4Ky2sSeLPh3ejd5kunXsTc8YWSIj/0M/pXn5or4Z/I9nIJWx0V3T/I+e/2H9ROl/tV/D25DON9/Nb/ACYz+9tpo/y+fn2zX7UV+G/7LF4LH9pD4aTNO0QbxPYw7gSM75VTbx67sfjX7kVhk7/dSXn+h18TR/2iEvL9WFFFFeufNhRRRQAUUUUAFFFFABRRRQBqab4n1/ScCx1WdFHRGben/fJyK6vTPi1ex4TV9OjnXoXhOxvrg5B/SuAornq4SjV+OJ24fMcVQ/hzdu26+5nt2l+OvDOq7Uj1BYJW/wCWdwPLP59D+BrfBBAIOQa+cq1NK8S65opH9najLGg/5Zk7kP8AwE8V5lbJ1vSl957uG4lktMRD5r/J/wCZ71RXnejfFiNtsWu2Gw9DNb8j8VPP5E/Su30zWNL1iHz9NvYp177T8y/UdR+NeVWwtWj8aPosLmGGxP8ACld9uv3F2iiiuc7AooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAoopGYKCzEAAZJPagBaq6hqdhpVubrUbuOCIfxOevsB1J9hXH+JvibZ2Jez0IJdTjgznmJfp/eP6fWvNdR1TUNXuTd6ldyTynux4A9AOgHsK9TDZZUqe9U0X4ngY/PqNC8KPvS/Bf5/L7zu9e+Krtut/D1ttHT7RMMn/gK/4/lXB32o32pzm51C7luJD/ABSNnHsPQVWor3KGFpUV7i/zPksVj8RiXerLTt0+4KKKK6DjCiiigAooooAKKKKACiiigAooooAKKKZcXEFrBJc3U8cMMSl5JJGCqgHUkngCgB9cz8SPiL4T+FHgvU/HnjXU0stL0uEySMSN8rfwxoP4nY4AXuTXhXxs/wCCgfwL+E8c+neH9UXxvryZVbPR5la2Rv8AppdcoB2ITewPVRX5o/Hr9o/4mftEeIU1nx1qaJZWhP8AZ+k2gKWdkD12qSSznu7EsemQAAPOxWY06StB3ke3l+SV8RJSqrlj+L9P8zn/AIw/EvVvjD8TfEXxK1qMRXOu3hnEIbcIIgAkUWe+yNEXPfbX1j/wSo8FXd/8T/Fvj94H+x6Poo0xXIIUzXEqOMHuQlu+R23D2r4u8OeHdc8Xa9YeGPDWmT6jquqXC2tpawLl5ZGOAB/ieAOTgV+137LvwIsf2evhHpvgdXin1aZjf6zdR8rPduAG2n+6oVUXgZCA4yTXl5bSlVr+1lstfme/nmIp4fCfV47y0S8l/Vj1uiiivpT4UKKKKACvwj/aB0s6J8d/iJpRUqtt4p1JE4Ayn2mTacD1XBr93K/E39s7Tl0v9qL4i2yhAH1b7T8vTMsaSH8fn5968fOF+7i/M+l4Zl+/nHy/U9+/4JQ3UafE3xvZHd5kugxSrxxhZ1B/9DH61+mlflx/wSpu1T44+KLEoS03hSWYN2AS6thj/wAfH5V+o9b5W/8AZl8zkz9f7dL0X5BRRRXonihRRRQB+R//AAUt0xbD9p67ugFB1HRLK5OCSThWi5z0P7rt7d81mf8ABOjUPsX7Vvhq287Z9vsr+327c78WskmOnH+rznjpjvXd/wDBVWyEfx08MagHH7/wnFEV29Nl1dHOe+d/6V5L+wteNYftX/D6dZ1iLXdzDubGDvtJ0xz3O7H48c18xU93H/8Aby/M+9pe/lH/AG4/wR+0VFFFfTnwQUUUUAFfC3/BVnwRdan8P/Bnj+2t2dNB1OfT7llXOxLlEZWY+ga3C/Vx61901y3xS+HWhfFr4fa78OvEiE2GuWjWzOFBaF+scqg/xI4Vx7qKwxNH2tKVPudeAxP1fEwrPZP8Nmfg1ousah4e1mw1/SLgwX2mXMd5aygZMcsbB0b8CAa/b79nX49+Fv2hfhxZeM9Cnhi1CNFh1jTQ+ZLC5x8ykddjYLI38S+4YD8ZPix8LfFvwa8ean8PvGdi1vf6dJhZAD5dzEfuTRn+JGHIPbkHBBAd8Kvi74/+C3iqHxh8PNel02+jGyVPvQ3MecmOWM8Op9D06gggGvncHipYWo1JadT7bM8vhj6KlTeq1T6O5+9NFfH/AMC/+Cknwl+IFvbaR8UgvgfX2wjSylpNNnb1WbGYs8nEmAOBvavrXSdY0jXrCHVtC1Sz1GxuF3RXNpOs0Ug9VdSQR9DX0lKvTqq9N3PhsRhK2Hly1otf13LdFFFanOFFFFABRRRQAUUUUAFFFFABRRRQAVLb3NxaTLcWs8kMq8q8bFSPxFRUUNX0Y02ndHdaF8UtRtCsGuQ/bIunmoAsg/Do36fWvRNI17Sddh8/TLxJcD5k6On1XqK8BqW1urmynW5tLiSGVDlXRiCK8zEZZSqaw91/ge5gs+xFH3avvx/H7/8AM+iaK848NfFDJSz8Rr7C6Rf/AEJR/MflXocFxBdQpcW0ySxSDcro2Qw9jXg18NUou1RH1+Ex1DFR5qT+XVElFFFYHYFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFUNa1qx0GwfUL+Tai8Ko+87dlA9acYuT5Y7kznGEXObskSapqtho1m99qNwsUSdz1Y+gHc15H4r8daj4iZraDda2GeIgfmk93Pf6dPr1rO8R+I7/AMSXxurtysa8RQg/LGP6n1NZNfSYLLo0vfqay/I+HzTOp4hulR0h+L/4Hl94UUUV6Z4IUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAV5p+0h8VtS+CfwW8S/EvR9HXU77SIYxbwSBjH5kkqRK74IOxTJuOCMgYyM5r0us7xHoGl+KdDvfD2tWcN1Y38RhngmjEkciHqrKeCD3BqZRck0nbzLpSjGalNXSeq7n4167+3F+1Lr+oSahL8WtRst7ErBYQQ28UY7KFVBkD1JJ9Sa8y8YfFP4mfEHC+OfiD4i19FO5I9S1Oa4RD/sq7EL+Ar9IvFv8AwTF+D2t38t7od9rehK7bhb2V8rwj6LNG7D/vvArn4f8Agl/8MrBzLqXiHxldopBAhubVVI75xCT+WK8SWVY2b5XJP/t7/M+ypZzlkFzQjyv/AA/5H5q16R8Hv2ePi58ddUWw+HvhO5urYPtn1OdTDY2/rvmI25/2Rlj2Br9GvB/7HP7PXgqVLi3+H1vqd3Gf9dq8r3mSPWNz5f8A45XtGnKmjxww6VGlnHbqEijgUIiKOgCjgD2rsocNzetaa9F/mc2J4oja2Hh83/kv8ziP2Wv2NPAv7OVmNbuJI9e8aXMWy51eSPC24I+aO3U/cXsW+83fA+UfQ9YWheJFvyLS82pP/Cw4D/4Gt2vQWHVBezSsfMVsRUrzdSq7thRRRTMgoorzz4vfH/4T/A3SX1P4ieLrSym2b4NPiYS3tz6COAfMQTxuOFHciplKMVzSdkXTpzqSUIK7Z2XiPxFovhHQdQ8T+JNRhsNL0u3e6u7mZsJFGoySfy6dT0Ffhb8bviK3xa+Lfiv4jeS8MWuanJcW8T/ejgB2xK3uI1QH3Br1b9qv9tDxn+0dcjw9YWsnh/wVazeZBpiy7pbtgflkuWHDEdQg+VSf4iA1fO1ra3N9cw2Vlby3FxcSLFFFEhd5HY4Cqo5JJIAAr5zMcYq7UIbL8T7jJcslhYurV+J/gj7o/wCCUPhe5uPiB448aeQfs9ho8Wl+aV4LzzCTaD9LbJ+o9a/S6vDv2OfgQ/wC+Cum+HdWgRPEWrOdV1sghtk8gAEWf+maBUOCRuDEda9xr2sDRdKhGMtz5XNcTHEYudSO2y+QUUUV1nnBRRRQB+bv/BWLwxcxeKPAPjNYmNvdWFzpjuM4VopFkUH0yJmx/un0r40+Efj6b4W/E7wx8Q4bY3H9ganDeyQDGZY1YeYgJ6Fk3AHsTmv2G/a5+Bh+P/wW1Twhp6R/27YuuqaKzkAfaowwCEnoHRnTJ4BYE9K/FO+sb3S76403UrSa1u7SVoLiCZCkkUikqysp5BBBBB6EV81mVOVPEe0XXU+6yKvCvg/YS3jdP0f9WP3+8KeKtA8ceG9N8XeFtTh1DSdWt1urS5iOVkRh+YI6EHkEEEAg1q1+Nf7LH7ZPjf8AZwvv7GuIpNe8GXUu+50iSXa1ux6y27HIRu5X7rd8H5h+pnwe/aH+Enx10xb/AOHviy2urkJvn0ydhDfW3rvhJ3Y/2hlT2Jr2MJjaddWvaXY+ZzHKq2Ek3a8ej/z7HpFFFFdp5YUUUUAeS/tEfs0fD39o/wAMLo/iyBrPVbJWOl6zbKPtFkx6jHR4zj5kbg9QVbDD8qvjv+yH8ZvgHdTXPiLQH1Tw+rfutd01WltSvbzON0J9nAGehbrX7Na3r0OlJ5SASXDDKp2X3NcXfXtzqTl72Uy5yNp+6B6YrKrk8cV77919z1MBnNbB+4vej2f6dj8Lq3PC3jvxt4Guje+C/GGtaDcE5Mmm38tszfUowz+Nfqx42/ZZ+Anj+WS6134cabFdyHc1zp+6ykLf3m8kqHP+8DXnF1/wTJ+FOrP52kav4ysAzZ2m6t2jA9t8O7053GvKq5BiqbvCSfzsfRU+I8HUjapFr5XR8eaX+2j+1HpE0M1r8ZtclMGNouvKuVOP7wkRg345r9L/ANif9oHxP+0P8JZ/EnjLTYLfWNH1N9KuLi2TZDebY45BIF/hbEmGUcZGRgNtHkeg/wDBLj4U2N2lzrHiDxFqcasD9nlvYoo2HcN5cQb8mFfXHw8+H3hn4Y+F7Xwj4S0q207TrXPlwW6bUUnqfUk9SxJJJJJrpweExNKXNWnp2vc8nNcdga1NQw8Pe72t/wAOdLRRRXonz4UUUUAFFFFABRRRQAUUUUAFFFFABRRRQAVt+G/FuqeGp82z+bbMcyW7n5W9x6H3/nWJRUTpxnHlmro0pVp0pqpTdmj3rQPEWm+IrQXVhL8w4kibh4z6Ef1rUr590rVr7Rb1L/T5zHKn5MO4I7ivaPC/iez8TWP2iH93PHgTwk8ofb1B7Gvm8dgHQ9+GsfyPucqzeOKXs6mk/wA/T/I2qKKK849sKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigCOeeK2hkuJ5AkcSl3Y9AB1NeI+LfE1x4l1NpyWW1iJW3jPZfU+5/8Ardq7P4qa81vaw6DbvhrkebPj+4DwPxIP5V5hX0GVYVRj7eW72PjeIce5T+qwei383/wPz9Aooor2D5kKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAp3+k2GpKRcwDd2deGH41yGsaBc6UfMB823JwHA6exFd3TZYo5o2ilQMjjDA9CK3pV5QfkJq55kCVIZSQRyCO1d7oGpnU7BXkOZojsk9/Q/j/jXGapZf2ffzWmSVRvlJ7g8j+davg2YpqEsGflkiz+II/xNduIip0uZER0Zn/HX4z+G/gH8N9Q+I3ia2uLuC0ZILe0t8CS5nc4RATwo6kk9ACcE8H4F1f/AIKufFiaeZtB+GfhKzhb/UpeSXNyyf7zK8Yb8AtfoB8bPhR4f+NXw61P4feJbd5LS+CurRvslikU5R0boGB5Gcg9CCCRX54+KP8AgmF8S7O8lXwl420m8t85j/tK1ntnAz0JjWUHA7jr6Dt83jKeNk08Pt5WufQ5TLLeRrF/Ffre1vl+p5x48/4KAftPeOopLRfHEXhy0lGGh0G1W1b0yJjumX8HH8q+fNR1LUdXvptT1a/uL28uG3zXFxK0ksjerMxJJ9zX1tp//BNX4pM6LrfjzwxZ/wB/7OtxMV57Bo0zx9P616Z4M/4Ju/D/AEyWO48ceOdX10pyYLOFLGFj6NkyOR9GU/TpXAsqzGu/fX3s95ZrleGjak18l/wD4K8L+FPEvjXXLbw14R0K+1jVLxtkFpZwtLI59cDoB1JPAHJxX6d/safsI23wgntviZ8WYrTUPGIUPYWCESwaQT/EW6ST9tw+VOdpY4YewfCzwN4G+DNi2mfDzwhpmlW0uBP5UWZp8dN8xzI5/wB5jXr2maraarD5lu2GX76H7y/59a9Kjkv1ZqpV95/gjwswz+eJi6VFcsX97/yLlFFFdp4IUUUUAFFFFABXx7+2R+wpZfGme5+JXwwa307xsUBvLSVhHbasFGAS2P3c2AAGPytgBsfeH2FVTUtUtdLh824b5j9xB95qzq4eNePs5K50YbFVcNUVWk7M/Ajxd4N8VeAteufC/jTw/faLqtocS2l5CY5B6EA9VPUMMgjkE1nWF/f6XeQ6jpl7PZ3duweKeCRo5I29VYEEH3Fftd8UvB3gr4yacuj/ABB8Jabq1lGSYVmi/ewk9Skow6E+qsK+V/Gf/BN/4darK9x4I8a6xoBfkQXUS30K+y8o4H1ZjXlVuHsTDWk0/wAH/l+J9ZhuJMNNWrrlf3r/AD/A+evAf7ff7TvgSBLIeOU8QWkahUh121W6YY9ZvllP4ua9X0j/AIKt/FqGeBtf+GnhG8hU/v0s3ubZnH+yzSSBfxDVz+of8E1figsjponj3wzdj/ln9oW4gLc9wqPjj61oeGP+CYfxOvLuJfFXjXSLO33Ayf2bbXF05Ge3mLEBx3P61nHC5pB8qT+9fqVVrZLNc0uX5Jr8j9BfgD8b/Dn7QXw3sviL4bsrmxSaV7W7s7ggvbXCY3puHDDDKQwxkMMgHIHe6nfJp1lLdsASg+Uep7CuI+BXwg8PfA34c2Hw/wDDUMi21s7zySTOHlnlfG6R2AALHA6AAAADgCtvxrORFbW4PDMzn8MAfzNe/hqcpOMKm/Wx8biHT9pJ0fhvp6HMTzS3MzzzOWdzliataXpF1qsuyEbY1+/Iei/4n2qrBE1xNHAn3pGCD6k4r0Wys4bC2S1gXCoOvcnuTXp163s42juc6Vyrp2g6fpwDJEJJR1kcZP4elaNFFeZKTk7yZoFFFFSAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABV/RNZvNB1GLUbJvmThlPR17qaoUUpRUk4y2ZUJyhJTg7NH0HpWp2usafDqVm2YplyPVT3B9weKt15Z8Ldea11CTQ53/dXeXiyfuyAc/mB+gr1OvkcXh3QquHTp6H6PluMWLw6q9dn6hRRRXMd4UUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAeGeNL9tR8T6hOTlUlMK/RPl/pn8axKnvmZ724dxhmlYkehyagr7WlFRgoroj8sr1HUqym+rbCiiirMgooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooA4XxQ2dbnGPuhR/46Kl8IrnV85+7Ex/lVXxA2/Wbo/7ePyAFXPB4J1VyO0LE/mK9WWlD5GfU7OiiivKNCOa3guU2XEKSL6MoNYeo+EbWYGSwfyH/uscof6iugoq4VJQ+FisebXdnc2Mxguoijj16EeoPeizvJ7G4S5t32up/Aj0PtXfanpsGp2zQTABuqP3U159PDJbzPBKMPGxVh7ivTo1lVjZkNWPRbC8i1C0ju4ujjkeh7ip65jwXcsRcWZPAxIv8j/Sunrza0OSbiWndBRRRWYwooooAhvbuKxtZLqY/LGM49T2Fee317PqFy1zcNlm6Dso9BXR+NLlljt7RTw5MjfhwP5muXhieeVIYxl5GCqPUmvSwlNKHO+pEn0H2lpcXswgtYi7nsO3ufSup07whbQgSahJ5z9dinCj+prV0rTINLthDEAXPMj92NXKwrYqUnaGiGokcFtb2y7LeBI19FUCpKKK5G77lBXKeNWzPap6Ix/Uf4V1dcd4yfOpRJ/dhH8zXThV+9QpbGboq7tWtBjP71TXodcD4cXfrVqP9on8lJrvqvGv30hR2CiiiuMoKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKALGn3kmnX9vfwn57eVZB74OcV9CRukqLKjZVwGB9Qa+c6+gdEJbRrBmJJNrGST/uivEzmKtCXqfV8MTd6kOmj/Mu0UUV4R9aFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFAHgviezNh4h1C1IwFuGK/wC6TkfoRWXXoHxW0VorqDXYU+SYCGYjsw+6fxHH4V5/X2GEqqrRjNH5pmOHdDFTpvvp6PYKKKK6DiCiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigDzrVn36pdt/02f+ZrU8GKf7SmbsICP/HlrFun8y5mf+9Ix/Wt3wWP9MuG9IsfrXrVtKLRmtzrqKKK8k0CiiigArgPEJB1m629Nw/kK7+vPNZffqt2f+mzD8jiuzBfGyZGl4MB/tKY9vIOf++lrsa5HwWD9tnbsIsfqK66oxf8VhHYKKKK5igooooA5DxmR9vgGeRDn9TWf4fUNrNqD/fz+QJq34wfdqqj+7Co/Un+tQeGF3a3bnHC7j/46a9WGlD5Gb+I7uiiivKNAooooAK4bxU+/WZV/uKq/pn+tdzXn2vP5msXTekm38uP6V14Ne+35Ey2LPhRd2sxn+6jH9Mf1ruK43wcudTkbssJ/mK7Kli3+8COwUUUVylBRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQA+GJ55kgiGXkYIo9STgV9DW0C21tFbJ92JAg+gGK8i+G+itqfiBLuRP3FgPOY44L/wAI/Pn/AIDXsVfP5vVTnGmun6n2XDWHcaU6z+1ovl/X4BRRRXjn0wUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAU9W0y21nTp9Nu1zHMuM91PYj3B5rwrWNJu9E1GbTb1MSRHgjo47MPY19A1z3jDwnb+JrH5dsd7CCYJT0/3W9j+n559HL8Z7GXLP4X+B4mc5Z9ap+0p/HH8V2/yPEqKnvLO60+6ks7yFopom2ujdQagr6ZNNXR8G007MKKKKYgooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACkdtiM5/hGaWoL9/LsbmT+7Ex/Q00ruwHm/Wuk8FLme6b0RR+p/wAK5quo8EgZvG7jYP8A0KvVxP8ACZnHc6iiiivJNAooooAK81vX8y8nk/vSsf1NelMQoLHoBmvMGJZix6k5Nd2CXxMiR0ngpc3F03oij9T/AIV1dcv4JA3XjdwEH/oVdRWOK/isqOwUUUVzjCiiigDhfFL7tamH90KP/HQf60/wmu7WFOPuxsf6f1qtrz79Yum9JMflxV7wcudUkbn5YD/Na9WWlD5GfU7KiiivKNAooooAK82vpPNvbiX+/KzfmTXo8r+XG8h/hUmvMiSTknJNd2CW7IkdH4LTNzcyeiAfmf8A61dZXM+CU+S7f1KD+f8AjXTVjin+9ZUdgooornGFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFS21tPeXEdrbRNJLKwRFHUk0yOOSaRYokZ3chVVRkknoAK9c8CeCl0KIanqKA6hKuApwRAD2Hue5/D1zy4vFRoQ5nv0R6GXZfUxlXkjst32/4JseFfD8XhzSI7FcNM37ydx/E56/gOgrYoor5Oc5Tk5y3Z+iUqUaUFTgrJBRRRUmgUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFAGF4p8JWHie2xL+5u4xiKcDkexHcV49rWhaloN2bTUYCjdUcco49Qe9e/VU1PS7DWLVrPUbZJom7MOVPqD1B9xXoYPMJ0PdlrH8vQ8XM8np4v8AeQ92ffv6/wCZ8+UV2fiX4balpZe60jfe2o52gfvUHuP4vqPyrjSCDg9a+jo1qdWPNTdz4nE4Wth58laNmJRRRWpzhRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAVS1t9mk3Z9YiPz4q7WZ4lfZotz6naP/HhV01eaXmDOCrq/BQ/dXZx1ZefzrlK67wWD9kuG7GQD9K9LFfwmZx3OioooryjQKKKKAIb1/Ls55M42xMf0Nea16JrT7NJu29YmH5jFed16OCXutkSOr8FD9zdNjqyj9DXS1z3gtf8AQ7hs9Zcfp/8AXroa5cT/ABWUtgooorAYUUUEgDJPAoA841F/M1C6kzndM5/U1s+C1zeXDekYH6//AFq5+RzI7OerEmul8Er8142egQf+hV61fSizNbnU0UUV5JoFFFFAFXVpPK0y6f0hbH1xXnNd74lk8vRbj1bav5sK4KvSwa9xsiR2HgxMWE8nrNj8gP8AGugrF8Ipt0nP96Vj/If0rarjru9RlLYKKKKxGFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAVZsNPvdUuksrC3eaaToqj9T6D3NbvhrwHq+vlLiVTaWZ586ReXH+yvf69K9W0Pw9pfh62+z6db7SfvyNy7n3P9Oledi8xp0fdjrL+tz2suyWtiWp1Pdh+L9P8zG8H+BbTw8q3t5tuNQI+9/DF7L7+/8AKuroor5yrVnVlzzd2fb4fD08PTVOkrJBRRRWZuFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABXPeIfA+ieIN0zxfZro8+fEMEn/aHRv5+9dDRV06k6cuaDszKtQp1o8lWN0eK694D17Q903k/a7ZefOhBOB/tL1H8veubr6Orn9b8D+H9c3SS2v2e4b/ltB8pJ9x0P4jNezh836Vl81/kfM4zhv7WFl8n/n/n954jRXY6z8MddsC0unMl/COyfLIP+Anr+BNclPbz2srQXMEkMi/eSRSrD8DXr0q9Oqr03c+axGEr4d2rRa/ruR0UUVqc4UUUUAFFFFABRRRQAUUUUAFFFFABWN4tfbpBH96RR/X+lbNc/wCM3xp8Kes2fyB/xrWgr1EJ7HH12Pgxf+JdM2es5H/jq1x1dp4PAGlMR3mYn8hXfi/4ZEdzcoooryzQKKKKAMzxK+zRbn1O0f8Ajwrgq7fxa+3SCP78ij+v9K4ivTwa/d/MiW52Xg1caZK2Os5/9BWt6sXwiMaRn1lY/wAq2q4a/wDEZS2CiiishhUN4/l2c8mcbI2b8hU1Utbfy9Ju29Yiv58f1qoK8kgPPK6zwUv7i6bPV1H6GuTrsfBi/wDEumb1mI/8dH+Nenin+6ZnHc36KKK8o0CiiigDD8YSbdLRP78wH6E1xddV41kxHaRerM35Y/xrla9XCq1JGctzvPDKbNFt89W3H/x41qVR0NNmkWg9Ywfz5q9Xm1HebfmWtgoooqBhRRRQAUUUUAFFFFABRRRQAUUUUAFFFORHkYJGpZmOAAMk0ANorqdH+HPiLVNsk8IsYT/FPw2PZOv54rvNE+HXh/Sdss8Rvpx/HOPlB9k6fnmuGvmFClpe78j1sJkuLxGvLyru/wDLc800PwjrniBg1laFYSeZ5fljH49/wzXpHh74daNo+y4vB9uul53SL8in2X+pz+FdWAFAAAAHAApa8XEZlWq6LReX+Z9Tgskw2H96XvS7v9EFFFFeeeyFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAUUUUAFFFFABRRRQAVU1DStN1WLydRsobhe29ASPoeo/CrdFNNp3RMoxkuWSujhdV+FGlXGZNKvJbRuyP8AvE/xH5muQ1P4e+J9Nywshdxj+O3O/wD8d+9+le00V30szrw0buvM8jEZFg6usVyvy/y2PnSWKWFzHNG0br1Vhgj8KZX0JfaZp2pJs1CxguBjA8yMNj6HtXM6j8L/AA5dkvaGeyc9o33L+TZP5EV6VLN6T/iJr8Tw6/DdeOtGSl+D/wAvxPIaK7jUPhRrMGW0+9t7pR2bMbH8OR+tc5feFvEWm5N5o9yqjqypvUf8CXIrup4qjU+CSPIrZfiqP8SDX5fejKopSCDg9aSug4wooooAKKKKACuZ8bPiO0T1Zz+WP8a6auT8avm5tk9EJ/M//Wrowq/eoUtjm67jwmMaOpx1kY/WuHru/C4I0WA+pY/+PGuvGfw/mTHc1aKKK8wsKKKKAOf8Zviwgj9Zs/kD/jXH11PjZ/ltI/UuT+lctXq4VfukZy3O68KrjRYjjG5mP15Na1ZvhtduiWwz2Y/mxrSrzqv8SXqWtgooorMYVleJ32aLOM8sVX/x4Vq1h+MH26Wi/wB+YD9Ca1oq9RCexxddt4RXbpOf70rH+Q/pXE13fhZduiwHj5ix/wDHiP6V3Yx/uyY7mrRRRXmFhRRRQByHjOTN9BFn7sW78yf8K56tjxXJv1mRf+eaKv6Z/rWR1r2KCtTRm9z0jT08uwtk/uwoP0FT0iKERUH8IApa8hu7uaBRRRSAKKKKACiiigAoorQsfD+t6nj7DpVzKp6OIyF/76PFTKUYq8nYqEJTfLBXfkZ9FdnYfCvxDc4a9ltrNe4Z97D8F4/Wul0/4VaFbYa/ubi8YdRny0P4Dn9a46mY4eH2r+h6lDJMbV+xZeen/B/A8oALEKoJJOAB3rd0zwP4m1XDQ6a8MZ/5aT/ux+R5P4CvYtP0PR9KAGnabBAR/EqDcfq3U1erz6ucPanH7z2cPwzFa1538l/m/wDI8+0r4TW0eJNZ1FpT3jgG1f8Avo8n8hXY6XoGj6Mu3TdPhhOMFwMufqx5NaFFebWxVar8cj3MNl+Gw/8ACgk++7+8KKKK5ztCiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAp3ujaTqIP2/Tbacnu8QJ/PrWDefDTwrdZMdtNak94ZT/JsiuqorWFerD4JNHPVwlCr/Egn8jzm8+EQ5aw1o+yzRf8AswP9Kxbr4YeKLcnyY7a5Hby5sf8AoWK9gorrhmmIju7+qPNq5Bgp7Jx9H/nc8HuvCniSzyZ9EuwB1KxlwPxGRWZJFJExSWNkYdQwwa+i6jlghnXZPCki+jqCP1rqhnMvtQ/E4KnDEH/DqNeqv/kfOtcb4xfdqka9lhH8zX1VceFPDV1nztDs8nqViCH8xiuc1f4MeAtYmNxLp9xBIwwWhuXH6EkV3YbO6EJ800zhq8NYlfBJP71+h8tV33htdui2wHox/wDHjXo13+zl4WkJNlreqQZ7SGOQD/x0Vat/gs1haR2tr4gWQRrgb7bbn8mNd1fOcHVikpfgzjeQY+L+C/o0cDRXcTfCbXlyYb+xkH+0zqf/AEE1Tl+GXiuM/JBbyf7sw/riudY3DvaaMZZXjY703+ZydFdDJ4B8XRfe0Zzxn5ZY2/k1VpPCPieP72hXh4z8sRb+VarEUntJfejGWDxMfipyXyZ5140fN3bx+kZP5n/61c5XY+L/AAt4pfU0ceGtVKCIKG+xSYPJJ5x71zr6BrsbFX0W/Vh1BtnBH6V7OHqQ9mrSRzSpVE9Yv7jstBXbo9qMY/d5q/UOk2lyml2im3lz5KfwH0FTV5s3eTCzW4UUUVIBXN+NZMQWsWfvOzfkB/jXSVyfjV83FrH/AHUZvzI/wrfDK9VClsc3Xf8Ah5dujWo4+6T+ZJrhlsrx1DLaTFSMgiM4Nej6NpeoDSrMLYXBzCp4ibuPpXTjZLkSuEISb0Q+ir6aBrsh2pot+x9BbOf6VPH4T8TS426FejP96Fl/nXlurTW8l95usPWe0H9zMmit+LwH4tmxs0WQZ/vOi/zNW4vhp4sk+/Zwx/706/0JqHiqC3mvvRrHL8XLanL7meJ65J5ur3bekpX8uP6VWtU8y6hT+9Io/WvU/wDhnrxnd3Es9xqekwiR2bHmyM3J9NmP1rW0z9nK7huYbm98VQgRSBysdoW3YIOMlhXoPN8FCNvaL8TSGS4+T0pv8F+bMGivUoPhJpK/8fOqXcn+4FT+YNaMHwz8KQ/6y2nn/wCukzD/ANBxXiSzXDra7+R2w4exst7L1f8Alc8cpVVmIVQST0Ar3W38H+F7XmLQ7Q4/56J5n/oWa04LW1tRttraKEekaBR+lYSzmH2Yv+vvOyHDFR/HUS9Ff/I8ItfDfiC8x9m0a8cH+LyWC/meK2LX4aeK7nHmWsNsD3lmH/suTXslFc083qv4UkdtPhrDr45N/cjzW0+EU5wb/Wo19VhiLfqSP5Vt2fwv8M22DcC5uj/00lwPyXFdfRXLPH4ie8vu0PQpZPgqe1NP11/MzrLw9oenYNlpNrEw6MIwW/76PNaNFFcspSk7ydz0IQjBWgrLyCiiipLCiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKACiiigAooooAKKKKAP/9k=';\n";
			source += "\n";
			source += "var doc = new jsPDF();\n";
			source += "\n";
			source += "doc.setFontSize(40);\n";
			source += "doc.text(\"Octocat loves jsPDF\", 40, 30, 4);\n";
			source += "doc.addImage(imgData, 10, 40, 180, 180);\n";
			editor.setValue(source);
			editor.gotoLine(0);
		});
	};

	var initAutoRefresh = function() {
		$('#auto-refresh').on('change', function() {
			if ($('#auto-refresh').is(':checked')) {
				$('.run-code').hide();
				jsPDFEditor.update();
			} else {
				$('.run-code').show();
			}
		});

		$('.run-code').click(function() {
			jsPDFEditor.update();
			return false;
		});
	};

	var initDownloadPDF = function() {
		$('.download-pdf').click(function(){
			eval('try{' + editor.getValue() + '} catch(e) { console.error(e.message,e.stack,e); }');

			var file = demos[$('#template').val()];
			if (file === undefined) {
				file = 'demo';
			}
			if (typeof doc !== 'undefined') {
				doc.save(file + '.pdf');
			} else if (typeof pdf !== 'undefined') {
				setTimeout(function() {
					pdf.save(file + '.pdf');
				}, 2000);
			} else {
				alert('Error 0xE001BADF');
			}
		});
		return false;
	};

	return {
		/**
		 * Start the editor demo
		 * @return {void}
		 */
		init: function() {

			// Init the ACE editor
			aceEditor();

			populateDropdown();
			loadSelectedFile();
			// Do the first update on init
			jsPDFEditor.update();

			initAutoRefresh();

			initDownloadPDF();
		},
		/**
		 * Update the iframe with current PDF.
		 *
		 * @param  {boolean} skipEval If true, will skip evaluation of the code
		 * @return
		 */
		update: function(skipEval) {
			setTimeout(function() {
				if (! skipEval) {
					eval('try{' + editor.getValue() + '} catch(e) { console.error(e.message,e.stack,e); }');
				}
				if (typeof doc !== 'undefined') try {
					
					if (navigator.appVersion.indexOf("MSIE") !==-1 || navigator.appVersion.indexOf("Edge") !==-1 || navigator.appVersion.indexOf('Trident') !== -1 ) {
						var options = {
							pdfOpenParams: {
								navpanes: 0,
								toolbar: 0,
								statusbar: 0,
								view: "FitV"
							},
							forcePDFJS: true,
							PDFJS_URL : 'examples/PDF.js/web/viewer.html'
						};
						PDFObject.embed(doc.output('bloburl'), "#preview-pane", options);
					} else {
						PDFObject.embed(doc.output('datauristring'), "#preview-pane");
					}
				} catch(e) {
					alert('Error ' + e);
				}
			}, 0);
		}
	};

}();

$(document).ready(function() {
	jsPDFEditor.init();
});
