
/* global describe, it, jsPDF, comparePdf, expect */
/**
 * Standard spec tests
 */

describe('Plugin: addimage getImageProperties', () => {
	  it('one blackpixel', () => {
		  var blackpixel = 'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQEAYABgAAD/2wBDAAIBAQIBAQICAgICAgICAwUDAwMDAwYEBAMFBwYHBwcGBwcICQsJCAgKCAcHCg0KCgsMDAwMBwkODw0MDgsMDAz/2wBDAQICAgMDAwYDAwYMCAcIDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAz/wAARCAABAAEDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREAAgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYkNOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOEhYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwD+f+iiigD/2Q==';
      var result = jsPDF.API.getImageProperties(blackpixel);
	    
	    expect(result.width).toEqual(1);
	    expect(result.height).toEqual(1);
	    expect(result.fileType).toEqual('JPEG');
	    expect(result.colorSpace).toEqual('DeviceRGB');
	    expect(result.bitsPerComponent).toEqual(8);
      
	  })
		
	  it('jpg', () => {
		  var jpg = 'data:image/jpeg;base64,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';
      var result = jsPDF.API.getImageProperties(jpg);
	    
	    expect(result.width).toEqual(280);
	    expect(result.height).toEqual(210);
	    expect(result.fileType).toEqual('JPEG');
	    expect(result.colorSpace).toEqual('DeviceRGB');
	    expect(result.bitsPerComponent).toEqual(8);
      
	  })
    
    it('colortype_1_grayscale_8_bit_png', () => {
		  var colortype_1_grayscale_8_bit_png = 'data:image/png;base64,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';
	    var result = jsPDF.API.getImageProperties(colortype_1_grayscale_8_bit_png);
	    
	    expect(result.width).toEqual(280);
	    expect(result.height).toEqual(210);
	    expect(result.fileType).toEqual('PNG');
	    expect(result.colorSpace).toEqual('DeviceGray');
	    expect(result.bitsPerComponent).toEqual(16);
	  })

})
